package mocks

import (
	"errors"

	"code.justin.tv/common/twirp"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/upload-service/transformations"
	"github.com/stretchr/testify/mock"
)

func SuccessfulUploadResponse() *uploader.UploadResponse {
	return &uploader.UploadResponse{
		Url:      "https://twitch-web-dev.s3-us-west-2.amazonaws.com/ttv-pictures-user-prod/uploaderino/?X-Amz-Algorithm=AWS4-HMAC-SHA256&X-Amz-Credential=AKIAIXJ6KWNCQLVN3D6A%2F20170506%2Fus-west-2%2Fs3%2Faws4_request&X-Amz-Date=20170506T004057Z&X-Amz-Expires=900&X-Amz-SignedHeaders=host&X-Amz-Signature=faff219819a638e798c82b814358f62d727ca43a8dd7e03470a62a6e4723568c",
		UploadId: "b6efdfa7-b866-4da4-9581-a7cd41fa0e85",
	}
}

func UnsuccessfulUploadResponseInvalidBucket() error {
	return twirp.InvalidArgumentError("OutputPrefix", "an error message")
}

func MockUploadServiceEmpty() *uploader.Empty {
	return &uploader.Empty{}
}

func MockSuccessUploadServiceStatusResponse() *uploader.StatusResponse {
	return &uploader.StatusResponse{
		Status:  uploader.Status_COMPLETE,
		Message: "",
	}
}

func MockTranscode() transformations.Transcode {
	return transformations.Transcode{
		Format:  "png",
		Quality: 100,
	}
}

func MockValidation() *uploader.Validation {
	return &uploader.Validation{
		FileSizeLessThan: "256KB",
	}
}

func MockValidOutput() *uploader.Output {
	xcode := MockTranscode()
	return &uploader.Output{
		Transformations: []*uploader.Transformation{
			xcode.AsProto(),
		},
		PostValidation: MockValidation(),
		Name:           "supercool-image-{{UploadID}}{{Extension}}",
	}
}

func MockInvalidOutput() *uploader.Output {
	return &uploader.Output{}
}

func MockCallback() *uploader.Callback {
	return &uploader.Callback{
		SnsTopicArn: "arn:aws:sns:mg-central-1:0000000000000:upload-service-callback",
		Data:        []byte{},
	}
}

func MockMonitoring() *uploader.Monitoring {
	return &uploader.Monitoring{}
}

func MockValidUploadRequest() uploader.UploadRequest {
	return uploader.UploadRequest{
		Outputs:       []*uploader.Output{MockValidOutput()},
		PreValidation: MockValidation(),
		OutputPrefix:  "lalala",
		Callback:      MockCallback(),
		Monitoring:    MockMonitoring(),
	}
}

func MockInvalidUploadRequest() uploader.UploadRequest {
	return uploader.UploadRequest{
		Outputs:       []*uploader.Output{MockInvalidOutput()},
		PreValidation: MockValidation(),
		OutputPrefix:  "lalala",
		Callback:      MockCallback(),
		Monitoring:    MockMonitoring(),
	}
}

func MockValidSetStatusRequest() uploader.SetStatusRequest {
	return uploader.SetStatusRequest{
		UploadId: "FakeId",
		Status:   uploader.Status_COMPLETE,
		Message:  "a message",
	}
}

func MockInvalidSetStatusRequest() uploader.SetStatusRequest {
	return uploader.SetStatusRequest{
		UploadId: "",
		Status:   uploader.Status_COMPLETE,
		Message:  "a message",
	}
}

func MockValidStatusRequest() uploader.StatusRequest {
	return uploader.StatusRequest{
		UploadId: "FakeId",
	}
}

func MockInvalidStatusRequest() uploader.StatusRequest {
	return uploader.StatusRequest{
		UploadId: "",
	}
}

func (m *Uploader) MockSuccessfulCreate() {
	r0 := SuccessfulUploadResponse()
	m.On("Create", mock.Anything, mock.Anything).Return(r0, nil)
}

func (m *Uploader) MockUnsuccessfulCreate() {
	m.On("Create", mock.Anything, mock.Anything).Return(nil, errors.New("test error"))
}

func (m *Uploader) MockUnsuccessfulInvalidBucketCreate() {
	m.On("Create", mock.Anything, mock.Anything).Return(nil, UnsuccessfulUploadResponseInvalidBucket())
}

func (m *Uploader) MockSuccessSetStatus() {
	r0 := MockUploadServiceEmpty()
	m.On("SetStatus", mock.Anything, mock.Anything).Return(r0, nil)
}

func (m *Uploader) MockFailureSetStatus() {
	m.On("SetStatus", mock.Anything, mock.Anything).Return(nil, errors.New("test error"))
}

func (m *Uploader) MockSuccessStatus() {
	r0 := MockSuccessUploadServiceStatusResponse()
	m.On("Status", mock.Anything, mock.Anything).Return(r0, nil)
}

func (m *Uploader) MockFailureStatus() {
	m.On("Status", mock.Anything, mock.Anything).Return(nil, errors.New("test error"))
}
