package server

import (
	"encoding/json"
	"fmt"
	"log"
	"reflect"
	"strconv"
	"strings"
	"testing"

	"code.justin.tv/web/upload-service/backend"
	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/upload-service/transformations"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/c2h5oh/datasize"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
)

type SerializationTestSuite struct {
	suite.Suite
	backend *backend.Backend

	testedFields map[string]struct{}
}

func (suite *SerializationTestSuite) SetupSuite() {
	logLevel := aws.LogOff
	// Uncomment logLevel for verbose logging from the aws clientlib.
	//logLevel = aws.LogDebugWithRequestErrors
	conf := &aws.Config{
		LogLevel:    &logLevel,                    // Whether or not to print debug messages
		DisableSSL:  aws.Bool(true),               // Moto cannot handle HTTPS
		Endpoint:    aws.String("localhost:5000"), // Connect to moto instead of amazon
		Credentials: credentials.NewStaticCredentials("a", "b", "c"),
		Region:      aws.String(backend.DefaultRegion)}
	be, err := backend.NewBackendFromConfigTableBucket(conf, "serialization_tableName", "s3://bucket/path", false)

	if err != nil {
		log.Fatal(err)
	}

	suite.backend = be

	// This field holds the names of all of the fields that have been tested during the tests
	suite.testedFields = map[string]struct{}{}
}

// TearDownSuite runs after all tests in the suite have been completed.
// The fields of uploader.UploadRequest are checked to ensure all fields have been tested
// during the tests. This is to prevent new fields being added without having been tested
func (suite *SerializationTestSuite) TearDownSuite() {
	t := reflect.TypeOf(uploader.UploadRequest{})

	// allFields is populated with the fields of uploader.UploadRequest
	allFields := map[string]struct{}{}
	collectFields(t.Name()+".", t, &allFields)

	foundZero := false
	for k, _ := range allFields {

		// Check if that field has been marked as tested during the tests
		if _, ok := suite.testedFields[k]; !ok {

			// Std out is shown if the test fails or if the test is ran with -v
			fmt.Printf("%s was not tested\n", k)
			foundZero = true
		}
	}

	// If fields have been missed, this will fail and the fields will have been printed above
	suite.Require().False(foundZero, "Not all fields were tested")
}

func (suite *SerializationTestSuite) SetupTest() {
	input := &dynamodb.CreateTableInput{
		AttributeDefinitions: []*dynamodb.AttributeDefinition{{
			AttributeName: aws.String("upload_id"),
			AttributeType: aws.String("S"),
		}},
		KeySchema: []*dynamodb.KeySchemaElement{{
			AttributeName: aws.String("upload_id"),
			KeyType:       aws.String("HASH"),
		}},
		ProvisionedThroughput: &dynamodb.ProvisionedThroughput{
			ReadCapacityUnits:  aws.Int64(5),
			WriteCapacityUnits: aws.Int64(5),
		},
		TableName: aws.String("serialization_tableName"),
	}
	_, err := suite.backend.DynamoDB().CreateTable(input)
	suite.Require().NoError(err)
}

func (suite *SerializationTestSuite) TearDownTest() {
	_, err := suite.backend.DynamoDB().DeleteTable(&dynamodb.DeleteTableInput{
		TableName: aws.String("serialization_tableName"),
	})
	suite.Require().NoError(err)
}

func (suite *SerializationTestSuite) TestMinimalRequest() {
	request := minimalRequest()

	upload := suite.SerializeRequest(request)
	suite.Require().Equal(request.Outputs[0].Name, upload.Outputs[0].Name)
	suite.Require().Equal(request.OutputPrefix, upload.OutputPrefix)
	suite.Require().Equal(request.Callback.SnsTopicArn, upload.Callback.ARN)
}

func (suite *SerializationTestSuite) TestMonitoring() {
	request := minimalRequest()
	request.Monitoring = &uploader.Monitoring{
		SnsTopic:      "sns topic",
		GrafanaPrefix: "grafana prefix",
		RollbarToken:  "rollbar token",
	}

	upload := suite.SerializeRequest(request)
	suite.Require().Equal(request.Monitoring.SnsTopic, upload.Monitoring.SNSTopic)
	suite.Require().Equal(request.Monitoring.GrafanaPrefix, upload.Monitoring.GrafanaPrefix)
	suite.Require().Equal(request.Monitoring.RollbarToken, upload.Monitoring.RollbarToken)
}

func (suite *SerializationTestSuite) TestCallback() {
	request := minimalRequest()
	request.Callback = &uploader.Callback{
		SnsTopicArn: "fakey-fake-ARN",
		Data:        []byte("feature service callback data"),
		PubsubTopic: "some.topic",
	}

	upload := suite.SerializeRequest(request)
	suite.Require().Equal(request.Callback.SnsTopicArn, upload.Callback.ARN)
	suite.Require().Equal(string(request.Callback.Data), string(upload.Callback.Data))
	suite.Require().Equal(request.Callback.PubsubTopic, upload.Callback.PubsubTopic)
}

func (suite *SerializationTestSuite) TestPrevalidation() {
	request := minimalRequest()
	request.PreValidation = &uploader.Validation{
		Format:           "JPG",
		FileSizeLessThan: "14 kb",
		AspectRatioConstraints: []*uploader.Constraint{
			{1.2, "="},
		},
		WidthConstraints: []*uploader.Constraint{
			{345, ">"},
			{6789, "<="},
		},
		HeightConstraints: []*uploader.Constraint{
			{344, ">="},
			{3404, "<"},
		},
	}

	upload := suite.SerializeRequest(request)
	suite.Require().Equal(strings.ToLower(request.PreValidation.Format), upload.PreValidation.Format)

	var size datasize.ByteSize
	err := size.UnmarshalText([]byte(request.PreValidation.GetFileSizeLessThan()))
	suite.Require().NoError(err)
	suite.Require().Equal(uint64(size), upload.PreValidation.FileSizeLessThan)

	structEquals(suite.T(), request, upload, "PreValidation", []string{
		"AspectRatioConstraints.[0].Value",
		"AspectRatioConstraints.[0].Test",
		"WidthConstraints.[0].Value",
		"WidthConstraints.[0].Test",
		"WidthConstraints.[1].Value",
		"WidthConstraints.[1].Test",
		"HeightConstraints.[0].Value",
		"HeightConstraints.[0].Test",
		"HeightConstraints.[1].Value",
		"HeightConstraints.[1].Test",
	})
}

func (suite *SerializationTestSuite) TestPrevalidationLegacy() {
	request := minimalRequest()
	request.PreValidation = &uploader.Validation{
		Format:           "JPG",
		FileSizeLessThan: "14 kb",
		AspectRatio:      195.0 / 209.0,
		MinimumSize: &uploader.Dimensions{
			Height: 205,
			Width:  190,
		},
		MaximumSize: &uploader.Dimensions{
			Height: 210,
			Width:  200,
		},
	}

	upload := suite.SerializeRequest(request)
	suite.Require().Equal(strings.ToLower(request.PreValidation.Format), upload.PreValidation.Format)

	var size datasize.ByteSize
	err := size.UnmarshalText([]byte(request.PreValidation.GetFileSizeLessThan()))
	suite.Require().NoError(err)
	suite.Require().Equal(uint64(size), upload.PreValidation.FileSizeLessThan)

	suite.Require().Equal(request.PreValidation.AspectRatio,
		upload.PreValidation.AspectRatioConstraints[0].Value)
	suite.Require().Equal("=", upload.PreValidation.AspectRatioConstraints[0].Test)
	suite.Require().Equal(float64(request.PreValidation.MinimumSize.Width),
		upload.PreValidation.WidthConstraints[0].Value)
	suite.Require().Equal(">=", upload.PreValidation.WidthConstraints[0].Test)
	suite.Require().Equal(float64(request.PreValidation.MaximumSize.Width),
		upload.PreValidation.WidthConstraints[1].Value)
	suite.Require().Equal("<=", upload.PreValidation.HeightConstraints[1].Test)
	suite.Require().Equal(float64(request.PreValidation.MinimumSize.Height),
		upload.PreValidation.HeightConstraints[0].Value)
	suite.Require().Equal(">=", upload.PreValidation.HeightConstraints[0].Test)
	suite.Require().Equal(float64(request.PreValidation.MaximumSize.Height),
		upload.PreValidation.HeightConstraints[1].Value)
	suite.Require().Equal("<=", upload.PreValidation.HeightConstraints[1].Test)
}

func (suite *SerializationTestSuite) TestOutputPostValidation() {
	request := minimalRequest()
	request.Outputs[0].PostValidation = &uploader.Validation{
		Format:           "JPG",
		FileSizeLessThan: "14 kb",
		AspectRatioConstraints: []*uploader.Constraint{
			{1.2, "="},
		},
		WidthConstraints: []*uploader.Constraint{
			{345, ">"},
			{6789, "<="},
		},
		HeightConstraints: []*uploader.Constraint{
			{344, ">="},
			{3404, "<"},
		},
	}

	upload := suite.SerializeRequest(request)
	suite.Require().Equal(strings.ToLower(request.Outputs[0].PostValidation.Format),
		upload.Outputs[0].PostValidation.Format)

	var size datasize.ByteSize
	err := size.UnmarshalText([]byte(request.Outputs[0].PostValidation.GetFileSizeLessThan()))
	suite.Require().NoError(err)
	suite.Require().Equal(uint64(size), upload.Outputs[0].PostValidation.FileSizeLessThan)

	structEquals(suite.T(), request, upload, "Outputs.[0].PostValidation", []string{
		"AspectRatioConstraints.[0].Value",
		"AspectRatioConstraints.[0].Test",
		"WidthConstraints.[0].Value",
		"WidthConstraints.[0].Test",
		"WidthConstraints.[1].Value",
		"WidthConstraints.[1].Test",
		"HeightConstraints.[0].Value",
		"HeightConstraints.[0].Test",
		"HeightConstraints.[1].Value",
		"HeightConstraints.[1].Test",
	})
}

func (suite *SerializationTestSuite) TestPostValidationLegacy() {
	request := minimalRequest()
	validation := &uploader.Validation{
		Format:           "JPG",
		FileSizeLessThan: "14 kb",
		AspectRatio:      195.0 / 209.0,
		MinimumSize: &uploader.Dimensions{
			Height: 205,
			Width:  190,
		},
		MaximumSize: &uploader.Dimensions{
			Height: 210,
			Width:  200,
		},
	}
	request.Outputs[0].PostValidation = validation

	upload := suite.SerializeRequest(request)
	postValidation := upload.Outputs[0].PostValidation

	suite.Require().Equal(strings.ToLower(validation.Format),
		postValidation.Format)

	var size datasize.ByteSize
	err := size.UnmarshalText([]byte(validation.GetFileSizeLessThan()))
	suite.Require().NoError(err)
	suite.Require().Equal(uint64(size), postValidation.FileSizeLessThan)

	suite.Require().Equal(validation.AspectRatio, postValidation.AspectRatioConstraints[0].Value)
	suite.Require().Equal("=", postValidation.AspectRatioConstraints[0].Test)
	suite.Require().Equal(float64(validation.MinimumSize.Width), postValidation.WidthConstraints[0].Value)
	suite.Require().Equal(">=", postValidation.WidthConstraints[0].Test)
	suite.Require().Equal(float64(validation.MaximumSize.Width), postValidation.WidthConstraints[1].Value)
	suite.Require().Equal("<=", postValidation.HeightConstraints[1].Test)
	suite.Require().Equal(float64(validation.MinimumSize.Height), postValidation.HeightConstraints[0].Value)
	suite.Require().Equal(">=", postValidation.HeightConstraints[0].Test)
	suite.Require().Equal(float64(validation.MaximumSize.Height), postValidation.HeightConstraints[1].Value)
	suite.Require().Equal("<=", postValidation.HeightConstraints[1].Test)
}

func (suite *SerializationTestSuite) TestOutputPermissions() {
	request := minimalRequest()
	request.Outputs[0].Permissions = &uploader.Permissions{
		GrantRead:        "uri=http://acs.amazonaws.com/groups/global/AllUsers",
		GrantFullControl: "id=ACCOUNT_ID",
	}

	upload := suite.SerializeRequest(request)
	suite.Require().Equal(request.Outputs[0].Permissions.GrantRead, upload.Outputs[0].GrantRead)
	suite.Require().Equal(request.Outputs[0].Permissions.GrantFullControl, upload.Outputs[0].GrantFullControl)
}

func (suite *SerializationTestSuite) TestAllTransformations() {

	transforms := []transformations.Transformation{
		&transformations.AspectRatio{2.0},
		&transformations.Crop{10, 11, 12, 13},
		&transformations.MaxHeight{14},
		&transformations.MaxWidth{15},
		&transformations.Transcode{Format: "png"},
		&transformations.Transcode{Format: "jpeg", Quality: 80},
		&transformations.ResizeDimensions{16, 17},
		&transformations.ResizeDimensions{Height: 18},
		&transformations.ResizeDimensions{Width: 19},
		&transformations.ResizePercentage{75},
	}
	protos := make([]*uploader.Transformation, len(transforms))
	for i, t := range transforms {
		protos[i] = t.AsProto()
	}

	request := minimalRequest()
	request.Outputs[0].Transformations = protos
	upload := suite.SerializeRequest(request)

	for i := 0; i < len(transforms); i++ {
		suite.Require().Equal(upload.Outputs[0].Transformations[i], transforms[i])
	}
}

func TestSerialization(t *testing.T) {
	suite.Run(t, new(SerializationTestSuite))
}

// SerializeRequest builds a models.Upload object from a given uploader.UploadRequest object,
// uploads it to dynamo, downloads it from dynamo, and returns the result. The object is checked
// after downloading to ensure it has not been modified during serialization. The suite additionally
// tracks which uploader.UploadRequest fields have been set over all tests
func (suite *SerializationTestSuite) SerializeRequest(request *uploader.UploadRequest) *models.Upload {

	upload, err := createUploadFromRequest(request)
	suite.Require().NoError(err)
	suite.backend.CreateMetadata(*upload)

	got, err := suite.backend.GetMetadata(upload.UploadId)
	suite.Require().NoError(err)

	uploadJSON, _ := json.Marshal(upload)
	gotJSON, _ := json.Marshal(got)
	suite.Require().True(reflect.DeepEqual(upload, got),
		"The upload to CreateMetadata was not retrieved identically from GetMetadata.\nUploaded:%s\nRetrieved:%s", uploadJSON, gotJSON)

	// This variable is filled in with all of the fields that have been set
	// in order to indicate which fields are being tested during the current run
	testedFields := map[string]struct{}{}

	val := reflect.Indirect(reflect.ValueOf(request))
	collectValues(val.Type().Name()+".", val, &testedFields)

	for k, _ := range testedFields {

		// The set of fields newly tested is added to the set of fields previously tested
		suite.testedFields[k] = struct{}{}
	}

	return got
}

// collectFields calculates the names of all fields in a given type, recursing through structs, pointers, and arrays.
// Interfaces are counted but not entered
func collectFields(prefix string, t0 reflect.Type, foundFields *map[string]struct{}) {
	for i := 0; i < t0.NumField(); i++ {
		name := prefix + t0.Field(i).Name
		t := t0.Field(i).Type

		for t.Kind() == reflect.Ptr || t.Kind() == reflect.Slice || t.Kind() == reflect.Array {
			t = t.Elem()
		}
		switch t.Kind() {
		case reflect.Struct:
			collectFields(name+".", t, foundFields)
		default:
			(*foundFields)[name] = struct{}{}
		}
	}
}

// collectValues calculates the names of all fields in a struct object which do not contain their zero value.
func collectValues(prefix string, val reflect.Value, testedFields *map[string]struct{}) {
	for i := 0; i < val.NumField(); i++ {
		name := prefix + val.Type().Field(i).Name
		v1 := val.Field(i)
		v2 := reflect.Zero(val.Field(i).Type())
		zero := reflect.DeepEqual(v1.Interface(), v2.Interface())
		if !zero {
			enterValue(name, v1, testedFields)
		}
	}
}

// enterValue checks for slices, structs, and arrays which can be entered recursively
func enterValue(name string, v1 reflect.Value, testedFields *map[string]struct{}) {
	for v1.Kind() == reflect.Ptr {
		v1 = v1.Elem()
	}
	switch v1.Kind() {
	case reflect.Slice:
		fallthrough
	case reflect.Array:
		for j := 0; j < v1.Len(); j++ {
			enterValue(name, v1.Index(j), testedFields)
		}
	case reflect.Struct:
		collectValues(name+".", v1, testedFields)
	default:
		(*testedFields)[name] = struct{}{}
	}
}

func minimalRequest() *uploader.UploadRequest {
	return &uploader.UploadRequest{
		Outputs: []*uploader.Output{
			{
				Name: "foo.png",
			},
		},
		OutputPrefix: "s3://not-a-real-account/not-a-real-bucket/uploaderino/",
		Callback: &uploader.Callback{
			SnsTopicArn: "fakey-fake-ARN",
		},
	}
}

// structEquals removes duplicate code for cases where the structs are from different packages but use
// the same naming structure. Prefix is a shortcut for calculating a particular field and passing it
// in as struct1 and struct2. You must have periods before array indices, i.e. 'field.[0]'
func structEquals(t *testing.T, struct1, struct2 interface{}, prefix string, paths []string) {
	for _, path := range paths {
		v1 := reflect.ValueOf(struct1)
		v2 := reflect.ValueOf(struct2)
		for _, next := range strings.Split(prefix+"."+path, ".") {
			v1 = followPath(t, v1, next)
			v2 = followPath(t, v2, next)
		}
		require.True(t, reflect.DeepEqual(v1.Interface(), v2.Interface()))
	}
}

// followPath calculates the value corresponding to the given field name OR array index
func followPath(t *testing.T, v1 reflect.Value, next string) reflect.Value {
	for v1.Kind() == reflect.Ptr {
		v1 = v1.Elem()
	}
	if strings.HasPrefix(next, "[") && strings.HasSuffix(next, "]") {
		index, err := strconv.Atoi(next[1 : len(next)-1])
		require.Nil(t, err)
		return v1.Index(index)
	} else {
		return v1.FieldByName(next)
	}
}
