package server

import (
	"net/url"
	"strings"
	"time"

	"fmt"

	"code.justin.tv/common/twirp"
	"code.justin.tv/web/upload-service/backend"
	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/upload-service/transformations"
	"github.com/c2h5oh/datasize"
	uuid "github.com/satori/go.uuid"
	"golang.org/x/net/context"
)

const MaxFileSizeDefault uint64 = 5 << 10 << 10 << 10

const invalidBucketPathMessage string = "You must specify a bucket and a path, i.e. s3://bucket/path"

type Server struct {
	backend backend.Backender
}

func NewServer(b backend.Backender) Server {
	return Server{
		backend: b,
	}
}

func (s *Server) Create(ctx context.Context, uploadRequest *uploader.UploadRequest) (newURL *uploader.UploadResponse, err error) {
	upload, err := createUploadFromRequest(uploadRequest)
	if err != nil {
		return nil, err
	}

	signedUrl, err := s.backend.CreatePresignedUrl(upload.UploadId)
	if err != nil {
		return nil, twirp.InternalErrorWith(err)
	} else {
		err := s.backend.CreateMetadata(*upload)

		if err != nil {
			return nil, twirp.InternalErrorWith(err)
		}
	}

	return &uploader.UploadResponse{
		Url:      signedUrl,
		UploadId: upload.UploadId,
	}, nil
}

func (s *Server) SetStatus(ctx context.Context, setStatusRequest *uploader.SetStatusRequest) (*uploader.Empty, error) {
	status := setStatusRequest.GetStatus()
	statusMessage := setStatusRequest.GetMessage()

	err := s.backend.SetStatus(ctx, setStatusRequest.GetUploadId(), status, statusMessage)
	if err != nil {
		return nil, twirp.InternalErrorWith(err)
	}

	return &uploader.Empty{}, nil
}

func (s *Server) Status(ctx context.Context, statusRequest *uploader.StatusRequest) (*uploader.StatusResponse, error) {
	uploadID := statusRequest.GetUploadId()

	upload, err := s.backend.GetMetadata(uploadID)
	if err != nil {
		if _, ok := err.(backend.UserError); ok {
			return nil, twirp.InvalidArgumentError("UploadID", err.Error())
		}
		return nil, twirp.InternalErrorWith(err)
	}

	resp := uploader.StatusResponse{
		UploadId: uploadID,
		Status:   uploader.Status(upload.StatusValue),
		Message:  upload.StatusMessage,
	}
	return &resp, nil
}

func createUploadFromRequest(request *uploader.UploadRequest) (*models.Upload, error) {
	if request.GetCallback().GetSnsTopicArn() == "" {
		return nil, twirp.RequiredArgumentError("Callback.SnsTopicArn")
	}
	requestedUrl, err := url.Parse(request.GetOutputPrefix())
	if err != nil {
		return nil, twirp.InvalidArgumentError("OutputPrefix", err.Error())
	}

	pathParts := strings.Split(requestedUrl.Path, "/")
	if len(pathParts) < 2 {
		return nil, twirp.InvalidArgumentError("OutputPrefix", invalidBucketPathMessage)
	}

	model := models.Upload{}
	model.StatusValue = int32(uploader.Status_REQUESTED)
	model.StatusName = uploader.Status_name[model.StatusValue]
	model.UploadId = uuid.NewV4().String()
	model.CreateTime = time.Now().Unix()

	model.Callback = models.UploadCallback{ARN: request.Callback.SnsTopicArn, Data: request.Callback.GetData(), PubsubTopic: request.Callback.PubsubTopic}
	if len(request.GetOutputs()) == 0 {
		return nil, twirp.InvalidArgumentError("Outputs", "At least one output is required")
	}
	model.Outputs = make([]models.Output, len(request.GetOutputs()))
	hasTransformations := false
	for i, output := range request.GetOutputs() {
		out, newTransformations, err := createOutputFromRequest(output, i)
		hasTransformations = hasTransformations || newTransformations
		if err != nil {
			return nil, err
		}
		model.Outputs[i] = out
	}
	model.OutputPrefix = request.OutputPrefix

	model.Monitoring.SNSTopic = request.Monitoring.GetSnsTopic()
	model.Monitoring.GrafanaPrefix = request.Monitoring.GetGrafanaPrefix()
	model.Monitoring.RollbarToken = request.Monitoring.GetRollbarToken()

	preValidation, err := convertValidation(request.GetPreValidation())
	if err != nil {
		return nil, twirp.InvalidArgumentError("Pre-Validation", err.Error())
	}
	if preValidation.FileSizeLessThan == 0 {
		preValidation.FileSizeLessThan = MaxFileSizeDefault
	}
	if hasTransformations && preValidation.Format == "" {
		preValidation.Format = "image"
	}
	model.PreValidation = preValidation

	return &model, nil
}

func createOutputFromRequest(output *uploader.Output, index int) (out models.Output, hasTransformations bool, e error) {
	out.Name = output.GetName()
	if out.Name == "" {
		e = twirp.RequiredArgumentError("Output.Name")
		return
	}

	if postValidation, err := convertValidation(output.GetPostValidation()); err != nil {
		e = twirp.InvalidArgumentError("Post-Validation", fmt.Sprintf("Output[%d] Post validation %s", index, err.Error()))
		return
	} else {
		out.PostValidation = postValidation
	}

	l := len(output.GetTransformations()) + 1
	out.Transformations = make([]transformations.Transformation, l)

	// always apply exif strip at the end
	out.Transformations[l-1] = &transformations.Strip{}

	for i, xf := range output.GetTransformations() {
		hasTransformations = true
		if x := xf.GetAspectRatio(); x != nil {
			out.Transformations[i] = &transformations.AspectRatio{x.GetRatio()}
			if x.GetRatio() <= 0 {
				e = twirp.InvalidArgumentError("Outputs", fmt.Sprintf("Output[%d] transformation[%d] illegal aspect ratio", index, i))
				return
			}
		} else if x := xf.GetCrop(); x != nil {
			out.Transformations[i] = &transformations.Crop{
				Top:    int(x.GetTop()),
				Left:   int(x.GetLeft()),
				Width:  uint(x.GetWidth()),
				Height: uint(x.GetHeight()),
			}
			if x.GetWidth() == 0 || x.GetHeight() == 0 {
				e = twirp.InvalidArgumentError("Outputs", fmt.Sprintf("Output[%d] transformation[%d] crop to zero size", index, i))
				return
			}
		} else if x := xf.GetMaxHeight(); x != nil {
			out.Transformations[i] = &transformations.MaxHeight{uint(x.GetHeight())}
			if x.GetHeight() == 0 {
				e = twirp.InvalidArgumentError("Outputs", fmt.Sprintf("Output[%d] transformation[%d] maximum zero height", index, i))
				return
			}
		} else if x := xf.GetMaxWidth(); x != nil {
			out.Transformations[i] = &transformations.MaxWidth{uint(x.GetWidth())}
			if x.GetWidth() == 0 {
				e = twirp.InvalidArgumentError("Outputs", fmt.Sprintf("Output[%d] transformation[%d] maximum zero width", index, i))
				return
			}
		} else if x := xf.GetResize(); x != nil {
			if d := x.GetDimensions(); d != nil {
				out.Transformations[i] = &transformations.ResizeDimensions{
					Width:  uint(d.GetWidth()),
					Height: uint(d.GetHeight()),
				}
			} else if d := x.GetPercent(); d != 0 {
				out.Transformations[i] = &transformations.ResizePercentage{Percent: uint(d)}
			} else {
				e = twirp.InvalidArgumentError("Outputs", fmt.Sprintf("Output[%d] transformation[%d].Resize invalid", index, i))
				return
			}
		} else if x := xf.GetTranscode(); x != nil {
			out.Transformations[i] = &transformations.Transcode{
				Format:      x.GetFormat(),
				Quality:     uint(x.GetQuality()),
				RemoveAlpha: x.GetRemoveAlpha(),
			}
		} else {
			e = twirp.InvalidArgumentError("Outputs", fmt.Sprintf("Output[%d] transformation[%d] unknown transformation", index, i))
			return
		}
	}

	out.GrantFullControl = output.GetPermissions().GetGrantFullControl()
	if out.GrantFullControl != "" && !validGrantString(out.GrantFullControl) {
		e = twirp.InvalidArgumentError("Outputs",
			fmt.Sprintf("Outputs[%d] invalid grant-full-control string %q", index, out.GrantFullControl))
		return
	}
	out.GrantRead = output.GetPermissions().GetGrantRead()
	if out.GrantRead != "" && !validGrantString(out.GrantRead) {
		e = twirp.InvalidArgumentError("Outputs",
			fmt.Sprintf("Outputs[%d] invalid grant-read string %q", index, out.GrantRead))
		return
	}

	return
}

func convertValidation(validation *uploader.Validation) (result models.Validation, retErr error) {
	if validation == nil {
		return
	}
	result = models.Validation{
		Format: strings.ToLower(validation.GetFormat()),
	}

	if validation.GetFileSizeLessThan() != "" {
		var size datasize.ByteSize
		err := size.UnmarshalText([]byte(validation.GetFileSizeLessThan()))
		if err != nil {
			return result, fmt.Errorf("Could not parse maximum file size: %q", validation.GetFileSizeLessThan())
		}
		result.FileSizeLessThan = size.Bytes()
		if result.FileSizeLessThan > MaxFileSizeDefault {
			return result, fmt.Errorf("Cannot specify a maximum file size larger than %s", datasize.ByteSize(MaxFileSizeDefault).String())
		}
	}

	result.AspectRatioConstraints = copyConstraints(validation.GetAspectRatioConstraints())
	if validation.GetAspectRatio() != 0 {
		result.AspectRatioConstraints = append(result.AspectRatioConstraints, models.Constraint{
			validation.GetAspectRatio(),
			"=",
		})
	}

	result.WidthConstraints = copyConstraints(validation.GetWidthConstraints())
	result.HeightConstraints = copyConstraints(validation.GetHeightConstraints())

	addDimensionConstraint(&result, validation.GetMinimumSize(), ">=")
	addDimensionConstraint(&result, validation.GetMaximumSize(), "<=")

	retErr = validConstraints(result.AspectRatioConstraints)
	if retErr == nil {
		retErr = validConstraints(result.WidthConstraints)
	}
	if retErr == nil {
		retErr = validConstraints(result.HeightConstraints)
	}

	return
}

func copyConstraints(constraints []*uploader.Constraint) (result []models.Constraint) {
	for _, constraint := range constraints {
		result = append(result, models.Constraint{
			constraint.GetValue(),
			constraint.GetTest(),
		})
	}
	return result
}

func addDimensionConstraint(result *models.Validation, dimension *uploader.Dimensions, test string) {
	if dimension != nil {
		result.WidthConstraints = append(result.WidthConstraints, models.Constraint{
			Value: float64(dimension.GetWidth()),
			Test:  test,
		})
		result.HeightConstraints = append(result.HeightConstraints, models.Constraint{
			Value: float64(dimension.GetHeight()),
			Test:  test,
		})
	}
}

func validConstraints(constraints []models.Constraint) error {
	if len(constraints) == 0 {
		return nil
	}
	countEq := 0
	seenGeGeq := []struct {
		float64
		bool
	}{}
	seenLeLeq := []struct {
		float64
		bool
	}{}
	seenInvalid := []string{}

	for _, constraint := range constraints {
		switch constraint.Test {
		case "=":
			countEq++
		case "<=":
			fallthrough
		case "<":
			seenLeLeq = append(seenLeLeq, struct {
				float64
				bool
			}{constraint.Value, constraint.Test == "<="})
		case ">=":
			fallthrough
		case ">":
			seenGeGeq = append(seenGeGeq, struct {
				float64
				bool
			}{constraint.Value, constraint.Test == ">="})
		default:
			seenInvalid = append(seenInvalid, constraint.Test)
		}
	}

	if len(seenInvalid) > 0 {
		return fmt.Errorf("Invalid constraint test(s): %v", seenInvalid)
	}

	oneGeOneLe := countEq == 0 && len(seenGeGeq) == 1 && len(seenLeLeq) == 1

	oneOnly := countEq+len(seenGeGeq)+len(seenLeLeq) == 1
	lineSegment := oneGeOneLe && seenGeGeq[0].float64 < seenLeLeq[0].float64
	pointIntersection := oneGeOneLe && seenGeGeq[0].float64 == seenLeLeq[0].float64 && seenGeGeq[0].bool && seenLeLeq[0].bool

	if oneOnly || lineSegment || pointIntersection {
		return nil
	} else {
		return fmt.Errorf("Redundant or mutually exclusive constraints")
	}
}

func validGrantString(grantString string) bool {
	for _, grantee := range strings.Split(grantString, ",") {
		noSpace := strings.Replace(grantee, " ", "", -1)
		if !strings.HasPrefix(noSpace, "id=") && !strings.HasPrefix(noSpace, "uri=") &&
			!strings.HasPrefix(noSpace, "emailaddress=") {
			return false
		}
	}
	return true
}
