package server

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"strings"
	"testing"

	"time"

	"code.justin.tv/common/twirp"
	"code.justin.tv/web/upload-service/backend"
	"code.justin.tv/web/upload-service/backend/mocks"
	"code.justin.tv/web/upload-service/models"
	"code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/upload-service/transformations"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/require"
	"github.com/stretchr/testify/suite"
)

type ServerTestSuite struct {
	suite.Suite
	TestServer *httptest.Server
	backender  mocks.Backender
}

func (suite *ServerTestSuite) SetupTest() {
	suite.backender = mocks.Backender{}

	rpcServer := NewServer(&suite.backender)
	rpcHandler := uploader.NewUploaderServer(&rpcServer, nil, nil)

	suite.TestServer = httptest.NewServer(rpcHandler)
}

func (suite *ServerTestSuite) TearDownTest() {
	suite.TestServer.Close()
}

func (suite *ServerTestSuite) TestValidUploadRequest() {
	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	suite.backender.On("CreatePresignedUrl", mock.AnythingOfType("string")).Return("s3://upload-service-ingest/THIS-IS-A-UUID", nil)
	suite.backender.On("CreateMetadata", mock.AnythingOfType("Upload")).Return(nil)

	request := basicRequest()

	url, err := client.Create(context.Background(), request)
	suite.Require().NoError(err)

	suite.Require().Equal(url.GetUrl(), "s3://upload-service-ingest/THIS-IS-A-UUID")
}

func (suite *ServerTestSuite) TestInvalidUploadRequestWithoutCallbackARN() {
	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	suite.backender.On("CreatePresignedUrl", mock.AnythingOfType("string")).Return("s3://upload-service-ingest/THIS-IS-A-UUID", nil)
	suite.backender.On("CreateMetadata", mock.AnythingOfType("Upload")).Return(nil)

	request := basicRequest()
	request.Callback.SnsTopicArn = ""

	_, err := client.Create(context.Background(), request)
	suite.Require().Error(err)
}

func (suite *ServerTestSuite) TestInvalidUploadRequestBadPaths() {
	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	_, expectedErr := url.Parse("::")
	request := basicRequest()
	request.OutputPrefix = "::"
	_, err := client.Create(context.Background(), request)

	// I believe this is the most reliable way to do this, even if it is slightly circuitous
	suite.Require().Error(err)
	suite.Require().True(strings.Contains(err.Error(), expectedErr.Error()), "'::' did not contain parse error for '::': %q", err.Error())

	// Currently we do not check for empty strings, meaning the following strings pass:
	// "s3:/", "s3:///", "s3://bucket/"
	// This is OK because we will check for permissions later, empty strings are better than non existing
	badStrings := []string{"s3:", "s3://", "s3://bucket"}
	for _, s := range badStrings {
		request := basicRequest()
		request.OutputPrefix = s
		_, err = client.Create(context.Background(), request)
		suite.Require().Error(err)
		suite.Require().True(strings.Contains(err.Error(), invalidBucketPathMessage),
			"'%s' did not return an error signaling missing bucket and/or path: %q", s, err.Error())
	}
}

func (suite *ServerTestSuite) TestInvalidUploadRequestFailedPresigning() {
	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	presigningErr := errors.New("example CreatePresignedUrl error")

	suite.backender.On("CreatePresignedUrl", mock.AnythingOfType("string")).Return("", presigningErr)
	suite.backender.On("CreateMetadata", mock.AnythingOfType("Upload")).Return(nil)

	request := basicRequest()

	_, err := client.Create(context.Background(), request)
	suite.Require().Error(err)
	suite.Require().True(strings.Contains(err.Error(), presigningErr.Error()), "err did not contain presigning error: %q", err.Error())

}

func (suite *ServerTestSuite) TestInvalidUploadRequestFailedCreateMetadata() {
	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	metadataErr := errors.New("example CreateMetadata error")

	suite.backender.On("CreatePresignedUrl", mock.AnythingOfType("string")).Return("s3://upload-service-ingest/THIS-IS-A-UUID", nil)
	suite.backender.On("CreateMetadata", mock.AnythingOfType("Upload")).Return(metadataErr)

	request := basicRequest()

	_, err := client.Create(context.Background(), request)
	suite.Require().Error(err)
	suite.Require().True(strings.Contains(err.Error(), metadataErr.Error()), "err did not contain metadata error: %q", err.Error())
}

func (suite *ServerTestSuite) TestCreateUploadLegalValidation() {
	output := &uploader.Output{
		Name: "foo.png",
		PostValidation: &uploader.Validation{
			FileSizeLessThan: "1",
			AspectRatioConstraints: []*uploader.Constraint{
				{1.2, "="},
			},
			WidthConstraints: []*uploader.Constraint{
				{345, ">"},
				{6789, "<="},
			},
		},
	}

	request := requestWithOutput(output)

	upload, err := createUploadFromRequest(request)
	suite.Require().NoError(err)
	validation := upload.Outputs[0].PostValidation

	suite.Require().Equal(uint64(1), validation.FileSizeLessThan)
	suite.Require().Equal(1, len(validation.AspectRatioConstraints))
	suite.Require().Equal(1.2, validation.AspectRatioConstraints[0].Value)
	suite.Require().Equal("=", validation.AspectRatioConstraints[0].Test)
	suite.Require().Equal(2, len(validation.WidthConstraints))
	suite.Require().Equal(345.0, validation.WidthConstraints[0].Value)
	suite.Require().Equal(">", validation.WidthConstraints[0].Test)
	suite.Require().Equal(6789.0, validation.WidthConstraints[1].Value)
	suite.Require().Equal("<=", validation.WidthConstraints[1].Test)
	suite.Require().Equal(0, len(validation.HeightConstraints))

	suite.Require().Equal(0.0, validation.AspectRatio)
	suite.Require().Nil(validation.MinimumSize)
	suite.Require().Nil(validation.MaximumSize)
}

func (suite *ServerTestSuite) TestCreateUploadPointIntersectionValidation() {
	output := &uploader.Output{
		Name: "foo.png",
		PostValidation: &uploader.Validation{
			FileSizeLessThan: "1",
			AspectRatioConstraints: []*uploader.Constraint{
				{1.2, "<="},
				{1.2, ">="},
			},
		},
	}

	request := requestWithOutput(output)

	upload, err := createUploadFromRequest(request)
	suite.Require().NoError(err)
	validation := upload.Outputs[0].PostValidation

	suite.Require().Equal(uint64(1), validation.FileSizeLessThan)
	suite.Require().Equal(2, len(validation.AspectRatioConstraints))
	suite.Require().Equal(1.2, validation.AspectRatioConstraints[0].Value)
	suite.Require().Equal("<=", validation.AspectRatioConstraints[0].Test)
	suite.Require().Equal(1.2, validation.AspectRatioConstraints[1].Value)
	suite.Require().Equal(">=", validation.AspectRatioConstraints[1].Test)
}

func (suite *ServerTestSuite) TestCreateUploadLegacyValidation() {
	output := &uploader.Output{
		Name: "foo.png",
		PostValidation: &uploader.Validation{
			FileSizeLessThan: "1",
			AspectRatio:      1.2,
			MinimumSize: &uploader.Dimensions{
				Height: 111,
				Width:  345,
			},
			MaximumSize: &uploader.Dimensions{
				Height: 2222,
				Width:  6789,
			},
		},
	}

	request := requestWithOutput(output)

	upload, err := createUploadFromRequest(request)
	suite.Require().NoError(err)
	validation := upload.Outputs[0].PostValidation

	suite.Require().Equal(uint64(1), validation.FileSizeLessThan)
	suite.Require().Equal(1, len(validation.AspectRatioConstraints))
	suite.Require().Equal(1.2, validation.AspectRatioConstraints[0].Value)
	suite.Require().Equal("=", validation.AspectRatioConstraints[0].Test)
	suite.Require().Equal(2, len(validation.WidthConstraints))
	suite.Require().Equal(345.0, validation.WidthConstraints[0].Value)
	suite.Require().Equal(">=", validation.WidthConstraints[0].Test)
	suite.Require().Equal(6789.0, validation.WidthConstraints[1].Value)
	suite.Require().Equal("<=", validation.WidthConstraints[1].Test)
	suite.Require().Equal(2, len(validation.HeightConstraints))
	suite.Require().Equal(111.0, validation.HeightConstraints[0].Value)
	suite.Require().Equal(">=", validation.HeightConstraints[0].Test)
	suite.Require().Equal(2222.0, validation.HeightConstraints[1].Value)
	suite.Require().Equal("<=", validation.HeightConstraints[1].Test)

	suite.Require().Equal(0.0, validation.AspectRatio)
	suite.Require().Nil(validation.MinimumSize)
	suite.Require().Nil(validation.MaximumSize)
}

func (suite *ServerTestSuite) TestCreateUploadCombineValidation() {
	output := &uploader.Output{
		Name: "foo.png",
		PostValidation: &uploader.Validation{
			FileSizeLessThan: "1",
			MinimumSize: &uploader.Dimensions{
				Height: 111,
				Width:  345,
			},
			WidthConstraints: []*uploader.Constraint{
				{6789, "<="},
			},
		},
	}

	request := requestWithOutput(output)

	upload, err := createUploadFromRequest(request)
	suite.Require().NoError(err)
	validation := upload.Outputs[0].PostValidation

	suite.Require().Equal(uint64(1), validation.FileSizeLessThan)
	suite.Require().Equal(0, len(validation.AspectRatioConstraints))
	suite.Require().Equal(2, len(validation.WidthConstraints))
	suite.Require().Equal(6789.0, validation.WidthConstraints[0].Value)
	suite.Require().Equal("<=", validation.WidthConstraints[0].Test)
	suite.Require().Equal(345.0, validation.WidthConstraints[1].Value)
	suite.Require().Equal(">=", validation.WidthConstraints[1].Test)
	suite.Require().Equal(1, len(validation.HeightConstraints))
	suite.Require().Equal(111.0, validation.HeightConstraints[0].Value)
	suite.Require().Equal(">=", validation.HeightConstraints[0].Test)

	suite.Require().Equal(0.0, validation.AspectRatio)
	suite.Require().Nil(validation.MinimumSize)
	suite.Require().Nil(validation.MaximumSize)
}

func (suite *ServerTestSuite) TestCreateUploadIllegalConstraintTest() {
	output := &uploader.Output{
		Name: "foo.png",
		PostValidation: &uploader.Validation{
			FileSizeLessThan: "1",
			WidthConstraints: []*uploader.Constraint{
				{344, "xyz"},
			},
		},
	}

	request := requestWithOutput(output)

	errString := "Invalid constraint test(s): [xyz]"

	_, err := createUploadFromRequest(request)
	suite.Require().Error(err)
	suite.Require().True(strings.Contains(err.Error(), errString), "Error did not contain %q: %q", errString, err.Error())
}

func (suite *ServerTestSuite) TestCreateUploadRedundantConstraint() {
	output := &uploader.Output{
		Name: "foo.png",
		PostValidation: &uploader.Validation{
			FileSizeLessThan: "1",
			HeightConstraints: []*uploader.Constraint{
				{344, ">"},
				{3404, "<"},
				{3144, "="},
			},
		},
	}

	request := requestWithOutput(output)

	_, err := createUploadFromRequest(request)
	suite.Require().Error(err)
	suite.Require().True(strings.Contains(err.Error(), "Redundant or mutually exclusive constraints"), "Error did not contain the correct message")
}

func (suite *ServerTestSuite) TestCreateUploadIllegalCombineValidation() {
	output := &uploader.Output{
		Name: "foo.png",
		PostValidation: &uploader.Validation{
			FileSizeLessThan: "1",
			MinimumSize: &uploader.Dimensions{
				Height: 111,
				Width:  345,
			},
			WidthConstraints: []*uploader.Constraint{
				{344, "<="},
			},
		},
	}

	request := requestWithOutput(output)

	_, err := createUploadFromRequest(request)
	suite.Require().Error(err)
	suite.Require().True(strings.Contains(err.Error(), "Redundant or mutually exclusive constraints"), "Error did not contain the correct message")
}

func (suite *ServerTestSuite) TestCreateAllTransforms() {
	transforms := []transformations.Transformation{
		&transformations.AspectRatio{2.0},
		&transformations.Crop{10, 11, 12, 13},
		&transformations.MaxHeight{14},
		&transformations.MaxWidth{15},
		&transformations.Transcode{Format: "png"},
		&transformations.Transcode{Format: "jpeg", Quality: 80},
		&transformations.ResizeDimensions{16, 17},
		&transformations.ResizeDimensions{Height: 18},
		&transformations.ResizeDimensions{Width: 19},
		&transformations.ResizePercentage{75},
	}

	// will be an exif strip added at the end
	transformsWithStrip := append(transforms, &transformations.Strip{})

	l := len(transforms)
	protos := make([]*uploader.Transformation, l)

	for i, t := range transforms {
		protos[i] = t.AsProto()
	}

	xformMatcher := func(upload models.Upload) bool {
		if len(upload.Outputs) != 1 {
			return false
		}
		return assert.ObjectsAreEqualValues(transformsWithStrip, upload.Outputs[0].Transformations)
	}

	suite.backender.On("CreatePresignedUrl", mock.AnythingOfType("string")).Return("s3://upload-service-ingest/THIS-IS-A-UUID", nil)
	suite.backender.On("CreateMetadata", mock.MatchedBy(xformMatcher)).Return(nil)

	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})
	request := requestWithOutput(&uploader.Output{
		Name:            "first",
		Transformations: protos,
	})

	_, err := client.Create(context.Background(), request)
	suite.Require().NoError(err)
}

func (suite *ServerTestSuite) TestValidGrantStrings() {
	// Test multiple grantees, and both full/read set, one set, neither set
	stringPairs := [][2]string{
		{
			"id = AWS_ACCOUNT_CANONICAL_USER_ID, uri = uri=http://acs.amazonaws.com/groups/global/AllUsers",
			"id = AWS_ACCOUNT_CANONICAL_USER_ID",
		},
		{
			"uri = uri=http://acs.amazonaws.com/groups/global/AllUsers",
			"id = AWS_ACCOUNT_CANONICAL_USER_ID, uri = uri=http://acs.amazonaws.com/groups/global/AllUsers",
		},
		{"id = AWS_ACCOUNT_CANONICAL_USER_ID, uri = uri=http://acs.amazonaws.com/groups/global/AllUsers", ""},
		{"", "id = AWS_ACCOUNT_CANONICAL_USER_ID, uri = uri=http://acs.amazonaws.com/groups/global/AllUsers"},
		{"id = AWS_ACCOUNT_CANONICAL_USER_ID", ""},
		{"", "uri = uri=http://acs.amazonaws.com/groups/global/AllUsers"},
		{"", ""},
	}

	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	suite.backender.On("CreatePresignedUrl", mock.AnythingOfType("string")).Return("s3://upload-service-ingest/THIS-IS-A-UUID", nil)
	suite.backender.On("CreateMetadata", mock.AnythingOfType("Upload")).Return(nil)

	for _, pair := range stringPairs {
		request := requestWithOutput(&uploader.Output{
			Name: "foo.png",
			Permissions: &uploader.Permissions{
				GrantFullControl: pair[0],
				GrantRead:        pair[1],
			},
		})

		url, err := client.Create(context.Background(), request)
		suite.Require().NoError(err)

		suite.Require().Equal(url.GetUrl(), "s3://upload-service-ingest/THIS-IS-A-UUID")
	}
}

func (suite *ServerTestSuite) TestInvalidGrantStrings() {
	badStrings := []string{
		"id = AWS_ACCOUNT_CANONICAL_USER_ID,", // Empty after comma
		",id = AWS_ACCOUNT_CANONICAL_USER_ID", // Empty before
		"id AWS_ACCOUNT_CANONICAL_USER_ID",    // No '='
	}

	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	suite.backender.On("CreatePresignedUrl", mock.AnythingOfType("string")).Return("s3://upload-service-ingest/THIS-IS-A-UUID", nil)
	suite.backender.On("CreateMetadata", mock.AnythingOfType("Upload")).Return(nil)

	for _, badString := range badStrings {
		request := requestWithOutput(&uploader.Output{
			Name: "foo.png",
			Permissions: &uploader.Permissions{
				GrantFullControl: badString, // GrantFullControl is checked first
				GrantRead:        badString,
			},
		})

		_, err := client.Create(context.Background(), request)
		suite.Require().Error(err)
		require.True(
			suite.T(),
			strings.Contains(err.Error(), "invalid grant-full-control string") && strings.Contains(err.Error(), badString),
			fmt.Sprintf("Error for %q did not contain %q and the string", badString, "invalid grant-full-control string"),
		)
	}
	for _, badString := range badStrings {
		request := requestWithOutput(&uploader.Output{
			Name: "foo.png",
			Permissions: &uploader.Permissions{
				GrantFullControl: "",
				GrantRead:        badString, // GrantRead is checked second
			},
		})

		_, err := client.Create(context.Background(), request)
		suite.Require().Error(err)
		suite.Require().True(strings.Contains(err.Error(), "invalid grant-read string") &&
			strings.Contains(err.Error(), badString), fmt.Sprintf("Error for %q did not contain %q and the string",
			badString, "invalid grant-read string"))
	}
}

func (suite *ServerTestSuite) TestValidStatusRequest() {
	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	request := &uploader.StatusRequest{
		UploadId: "fakeId",
	}

	mockUpload := models.Upload{
		UploadId:    "fakeId",
		StatusValue: 0,
		StatusName:  "REQUESTED",
		CreateTime:  int64(time.Now().Unix()),
	}

	suite.backender.On("GetMetadata", "fakeId").Return(&mockUpload, nil)

	resp, err := client.Status(context.Background(), request)
	suite.Require().NoError(err)

	suite.Require().Equal(resp.GetStatus(), uploader.Status_REQUESTED)
}

func (suite *ServerTestSuite) TestInvalidStatusRequest400() {
	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	request := &uploader.StatusRequest{}

	errMsg := "Get metadata received empty upload id"

	suite.backender.On("GetMetadata", "").Return(nil, backend.UserError{errors.New(errMsg)})

	_, err := client.Status(context.Background(), request)
	suite.Require().Error(err)
	suite.Require().True(strings.Contains(err.Error(), errMsg), "Error did not contain %q: %q", errMsg, err.Error())

	twirpErr, ok := err.(twirp.Error)
	suite.Require().True(ok, "Error was not twirp error")
	suite.Require().Equal(400, twirp.ServerHTTPStatusFromErrorCode(twirpErr.Code()))
}

func (suite *ServerTestSuite) TestInvalidStatusRequest500() {
	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	request := &uploader.StatusRequest{}

	errMsg := "Missing required config argument: METADATA_TABLE"

	suite.backender.On("GetMetadata", "").Return(nil, errors.New(errMsg))

	_, err := client.Status(context.Background(), request)
	suite.Require().Error(err)
	suite.Require().True(strings.Contains(err.Error(), errMsg), "Error did not contain %q: %q", errMsg, err.Error())

	twirpErr, ok := err.(twirp.Error)
	suite.Require().True(ok, "Error was not twirp error")
	suite.Require().Equal(500, twirp.ServerHTTPStatusFromErrorCode(twirpErr.Code()))
}

func (suite *ServerTestSuite) TestRequestStatusMessage() {
	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	id := "fakeId"

	request := &uploader.StatusRequest{
		UploadId: id,
	}

	statusMessage := "Error while publishing status to SNS"

	mockUpload := models.Upload{
		UploadId:      id,
		StatusValue:   5,
		StatusName:    "POSTPROCESS_FAILED",
		CreateTime:    int64(time.Now().Unix()),
		StatusMessage: statusMessage,
	}

	suite.backender.On("GetMetadata", "fakeId").Return(&mockUpload, nil)

	resp, err := client.Status(context.Background(), request)
	suite.Require().NoError(err)

	suite.Require().Equal(resp.GetStatus(), uploader.Status_POSTPROCESS_FAILED)
	suite.Require().Equal(resp.GetMessage(), statusMessage)
}

func (suite *ServerTestSuite) TestSetStatusToFailed() {
	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	setRequest := &uploader.SetStatusRequest{
		UploadId: "fakeId",
		Status:   uploader.Status_POSTPROCESS_FAILED,
		Message:  "Error while publishing status to SNS",
	}

	suite.backender.On("SetStatus", mock.Anything, mock.AnythingOfType("string"), mock.AnythingOfType("uploader.Status"),
		mock.AnythingOfType("string")).Return(nil)

	_, err := client.SetStatus(context.Background(), setRequest)
	suite.Require().NoError(err)
}

func (suite *ServerTestSuite) TestInvalidSetStatusRequest() {
	client := uploader.NewUploaderProtobufClient(suite.TestServer.URL, &http.Client{})

	setRequest := &uploader.SetStatusRequest{
		UploadId: "fakeId",
		Status:   uploader.Status_POSTPROCESS_FAILED,
		Message:  "Error while publishing status to SNS",
	}

	errMsg := "error message"

	suite.backender.On("SetStatus", mock.Anything, mock.AnythingOfType("string"), mock.AnythingOfType("uploader.Status"),
		mock.AnythingOfType("string")).Return(errors.New(errMsg))

	_, err := client.SetStatus(context.Background(), setRequest)
	suite.Require().Error(err)
	suite.Require().True(strings.Contains(err.Error(), errMsg), "Error did not contain %q: %q", errMsg, err.Error())
}

func (suite *ServerTestSuite) TestCreateUploadInvalidMaxFileSize() {
	request := requestWithOutput(&uploader.Output{
		Name: "foo.png",
		PostValidation: &uploader.Validation{
			FileSizeLessThan: fmt.Sprintf("%d b", MaxFileSizeDefault+1),
		},
	})

	errMsg := "Cannot specify a maximum file size larger than"

	_, err := createUploadFromRequest(request)
	suite.Require().Error(err)
	suite.Require().True(strings.Contains(err.Error(), errMsg), "Error did not contain %q: %q", errMsg, err.Error())
}

// See api/api_test.go for a test of SetStatusSNS-SubscriptionConfirmation

func TestServer(t *testing.T) {
	suite.Run(t, new(ServerTestSuite))
}

func basicRequest() *uploader.UploadRequest {
	return requestWithOutput(&uploader.Output{
		Name: "foo.png",
	})
}

func requestWithOutput(output *uploader.Output) *uploader.UploadRequest {
	return &uploader.UploadRequest{
		Outputs:      []*uploader.Output{output},
		OutputPrefix: "s3://not-a-real-account/not-a-real-bucket/uploaderino/",
		Callback:     &uploader.Callback{SnsTopicArn: "fakey-fake-ARN"},
	}
}
