data "aws_iam_policy_document" "eb_role" {
  statement {
    effect = "Allow"
    principals {
      type = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
    actions = ["sts:AssumeRole"]
  }
}

resource "aws_iam_role" "eb" {
  name = "${var.common_name}"
  path = "/"

  assume_role_policy = "${data.aws_iam_policy_document.eb_role.json}"
}

data "aws_iam_policy_document" "eb_role_policy" {
  # This statement gives access to EB- and Service-related s3
  statement {
    actions = [
      "s3:Get*",
      "s3:List*",
      "s3:PutObject*",
    ]

    effect = "Allow"
    resources = [
      "arn:aws:s3:::elasticbeanstalk-*",
      "arn:aws:s3:::elasticbeanstalk-*/*",
      "arn:aws:s3:::${var.s3_bucket}"
    ]
  }

  # Access for the service's cloudwatch-logs streams
  statement {
    actions = ["logs:*"]
    effect = "Allow"
    resources = [
      "arn:aws:logs:*:*:log-group:/aws/elasticbeanstalk/${var.common_name}*",
    ]
  }

  # Access for the service's dynamodb table
  statement {
    actions = ["dynamodb:*"],
    effect = "Allow"
    resources = ["arn:aws:dynamodb:*:*:table/${var.metadata_table}"]
  }

  # Allow the role to publish to any SNS Topic and PutObject(Acl) to any S3 bucket
  # This enables the service to copy uploads to output buckets and publish callbacks without
  # any further configuration from us
  statement {
    actions = ["s3:PutObject*", "sns:Publish"]
    effect = "Allow"
    resources = ["arn:aws:sns:*", "arn:aws:s3:::*"]
  }
}

resource "aws_iam_role_policy" "eb" {
  name   = "${var.common_name}"
  policy = "${data.aws_iam_policy_document.eb_role_policy.json}" 
  role   = "${aws_iam_role.eb.id}"
}

resource "aws_iam_instance_profile" "eb" {
  depends_on = ["aws_iam_role.eb"]
  name       = "${var.common_name}"
  role      = "${aws_iam_role.eb.name}"
}

resource "aws_iam_role_policy_attachment" "eb_worker_tier" {
    role       = "${aws_iam_role.eb.name}"
    policy_arn = "arn:aws:iam::aws:policy/AWSElasticBeanstalkWorkerTier"
}

resource "aws_iam_role_policy_attachment" "eb_web_tier" {
    role       = "${aws_iam_role.eb.name}"
    policy_arn = "arn:aws:iam::aws:policy/AWSElasticBeanstalkWebTier"
}

resource "aws_iam_role_policy_attachment" "eb_enhanced_health" {
    role       = "${aws_iam_role.eb.name}"
    policy_arn = "arn:aws:iam::aws:policy/service-role/AWSElasticBeanstalkEnhancedHealth"
}

data "aws_iam_policy_document" "pub_to_callback" {
  statement {
    actions = ["sns:Publish"]
    effect = "Allow"
    resources = ["${aws_sns_topic.callback.arn}"]
    principals = {
      type = "AWS"
      identifiers = ["${aws_iam_role.eb.arn}"]
    }
  }
}

resource "aws_sns_topic_policy" "pub_to_callback" {
  arn = "${aws_sns_topic.callback.arn}"

  policy = "${data.aws_iam_policy_document.pub_to_callback.json}"
}

data "aws_iam_policy_document" "sqs_senders" {
  statement {
    effect = "Allow"
    principals {
      type = "*"
      identifiers = ["*"]
    }
    actions = ["sqs:SendMessage"]
    resources = ["${var.sqs_queue_arn}"]
    condition {
      test = "ArnEquals"
      variable = "aws:SourceArn"
      values = ["${aws_sns_topic.callback.arn}"]
    }
  }

  statement {
    effect = "Allow"
    principals {
      type = "*"
      identifiers = ["*"]
    }
    actions = ["sqs:SendMessage"]
    resources = ["${var.sqs_queue_arn}"]
    condition {
      test = "ArnEquals"
      variable = "aws:SourceArn"
      values = ["${var.s3_bucket_arn}"]
    }
  }
}

resource "aws_sqs_queue_policy" "sqs_senders_attachment" {
  queue_url = "${var.sqs_queue_url}"
  policy = "${data.aws_iam_policy_document.sqs_senders.json}"
}

data "aws_iam_policy_document" "s3-ingest" {
  statement {
    effect = "Allow"
    principals {
      type = "AWS"
      identifiers = ["${aws_iam_role.eb.arn}"]
    }
    actions = ["s3:*"]
    resources = ["${var.s3_bucket_arn}/*"]
  }
}

resource "aws_s3_bucket_policy" "ingest-s3-policy-attachment" {
  bucket = "${var.s3_bucket}"
  policy = "${data.aws_iam_policy_document.s3-ingest.json}"
}
