resource "aws_elastic_beanstalk_environment" "server" {
  name         = "${var.common_name}-server-env"
  description  = "${var.common_name}"
  application  = "${var.eb_application_name}"
  tier         = "WebServer"
  solution_stack_name = "${var.solution_stack_name}"

  tags {
    Name        = "${var.common_name}-server"
    Environment = "${var.env}"
    Service     = "${var.service}"
    Owner       = "${var.owner}"
    Tier        = "WebServer"
    Skeleton    = "dta/docker-eb-tf-skeleton@503764814949793d49d5e873510d1da36014b583"
  }

  wait_for_ready_timeout = "${var.wait_for_ready_timeout}"

  # Load Balancer Related Settings
  setting {
    namespace = "aws:ec2:vpc"
    name      = "ELBSubnets"
    value     = "${var.elb_subnet_ids}"
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "ELBScheme"
    value     = "${var.elb_scheme}"
  }

  setting {
    namespace = "aws:elb:loadbalancer"
    name      = "SecurityGroups"
    value     = "${var.elb_loadbalancer_security_groups}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application"
    name = "Application Healthcheck URL"
    value = "${lookup(var.healthcheck_url, "server")}"
  }

  setting {
    namespace = "aws:elb:policies"
    name = "ConnectionDrainingEnabled"
    value = "true"
  }

  setting {
    namespace = "aws:elb:policies"
    name = "ConnectionDrainingTimeout"
    value = 10
  }

  # asg launch configuration settings
  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "IamInstanceProfile"
    value     = "${coalesce(var.auto_scaling_lc_iam_instance_profile, aws_iam_instance_profile.eb.name)}"
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name = "MinSize"
    value = "${lookup(var.asg_min_size, "server")}"
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name = "MaxSize"
    value = "${lookup(var.asg_max_size, "server")}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "InstanceType"
    value     = "${lookup(var.instance_type, "server")}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "RootVolumeType"
    value     = "${lookup(var.root_volume_type, "server")}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "RootVolumeSize"
    value     = "${lookup(var.root_volume_size, "server")}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "RootVolumeIOPS"
    value     = "${lookup(var.root_volume_iops, "server")}"
  }

  # eb environment settings
  setting {
    namespace = "aws:elasticbeanstalk:environment"
    name      = "ServiceRole"
    value     = "${var.eb_environment_service_role}"
  }

  # Vpc  and network related settings
  setting {
    namespace = "aws:ec2:vpc"
    name      = "VPCId"
    value     = "${var.vpc_id}"
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "Subnets"
    value     = "${var.ec2_subnet_ids}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "MonitoringInterval"
    value     = "${var.auto_scaling_lc_monitoring_interval}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "SecurityGroups"
    value     = "${var.auto_scaling_lc_security_groups}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "APP"
    value     = "${var.eb_application_name}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "STATSD_HOST_PORT"
    value     = "${var.statsd_host}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "ENVIRONMENT"
    value     = "${var.env}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "PUBSUB_ENDPOINT"
    value     = "${var.pubsub_endpoint}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "TOPIC_PREFIX"
    value     = "${var.topic_prefix}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "S3_BUCKET"
    value     = "${var.s3_bucket}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "METADATA_TABLE"
    value     = "${var.metadata_table}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "LOG_LEVEL"
    value     = "${var.log_level}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:environment"
    name      = "ServiceRole"
    value     = "${var.eb_environment_service_role}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name = "DeploymentPolicy"
    value = "rolling"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name = "BatchSize"
    value = "34"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name = "BatchSizeType"
    value = "Percentage"
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name = "RollingUpdateType"
    value = "Health"
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name = "MaxBatchSize"
    value = "1"
  }
  
  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name = "MinInstancesInService"
    value = "1"
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name = "SystemType"
    value = "enhanced"
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name = "ConfigDocument"
    value = "{\"Version\":1,\"CloudWatchMetrics\":{\"Instance\":{\"RootFilesystemUtil\":60,\"CPUIrq\":60,\"LoadAverage5min\":60,\"ApplicationRequests5xx\":60,\"ApplicationRequests4xx\":60,\"CPUUser\":60,\"LoadAverage1min\":60,\"ApplicationLatencyP50\":60,\"CPUIdle\":60,\"InstanceHealth\":60,\"ApplicationLatencyP95\":60,\"ApplicationLatencyP85\":60,\"ApplicationLatencyP90\":60,\"CPUSystem\":60,\"ApplicationLatencyP75\":60,\"CPUSoftirq\":60,\"ApplicationLatencyP10\":60,\"ApplicationLatencyP99\":60,\"ApplicationRequestsTotal\":60,\"ApplicationLatencyP99.9\":60,\"ApplicationRequests3xx\":60,\"ApplicationRequests2xx\":60,\"CPUIowait\":60,\"CPUNice\":60},\"Environment\":{\"InstancesSevere\":60,\"InstancesDegraded\":60,\"ApplicationRequests5xx\":60,\"ApplicationRequests4xx\":60,\"ApplicationLatencyP50\":60,\"ApplicationLatencyP95\":60,\"ApplicationLatencyP85\":60,\"InstancesUnknown\":60,\"ApplicationLatencyP90\":60,\"InstancesInfo\":60,\"InstancesPending\":60,\"ApplicationLatencyP75\":60,\"ApplicationLatencyP10\":60,\"ApplicationLatencyP99\":60,\"ApplicationRequestsTotal\":60,\"InstancesNoData\":60,\"ApplicationLatencyP99.9\":60,\"ApplicationRequests3xx\":60,\"ApplicationRequests2xx\":60,\"InstancesOk\":60,\"InstancesWarning\":60}}}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:cloudwatch:logs"
    name = "StreamLogs"
    value = "true"
  }
}

resource "aws_elastic_beanstalk_environment" "worker" {
  name         = "${var.common_name}-worker-env"
  description  = "${var.common_name}"
  application  = "${var.eb_application_name}"
  tier         = "Worker"
  solution_stack_name = "${var.solution_stack_name}"

  tags {
    Name        = "${var.common_name}-worker"
    Environment = "${var.env}"
    Service     = "${var.service}"
    Owner       = "${var.owner}"
    Tier        = "Worker"
    Skeleton    = "dta/docker-eb-tf-skeleton@503764814949793d49d5e873510d1da36014b583"
  }

  wait_for_ready_timeout = "${var.wait_for_ready_timeout}"

  # asg launch configuration settings
  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "IamInstanceProfile"
    value     = "${coalesce(var.auto_scaling_lc_iam_instance_profile, aws_iam_instance_profile.eb.name)}"
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name = "MinSize"
    value = "${lookup(var.asg_min_size, "worker")}"
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name = "MaxSize"
    value = "${lookup(var.asg_max_size, "worker")}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "InstanceType"
    value     = "${lookup(var.instance_type, "worker")}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "RootVolumeType"
    value     = "${lookup(var.root_volume_type, "worker")}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "RootVolumeSize"
    value     = "${lookup(var.root_volume_size, "worker")}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "RootVolumeIOPS"
    value     = "${lookup(var.root_volume_iops, "worker")}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application"
    name = "Application Healthcheck URL"
    value = "${lookup(var.healthcheck_url, "worker")}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:sqsd"
    name = "WorkerQueueURL"
    value = "${var.worker_queue_url}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:sqsd"
    name = "HttpPath"
    value = "/worker"
  }

  setting {
    namespace = "aws:elasticbeanstalk:sqsd"
    name = "ConnectTimeout"
    value = "1"
  }

  setting {
    namespace = "aws:elasticbeanstalk:sqsd"
    name = "InactivityTimeout"
    value = "${var.sqsd_timeout}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:sqsd"
    name = "VisibilityTimeout"
    value = "${var.sqsd_timeout}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name = "SQSD_TIMEOUT"
    value = "${var.sqsd_timeout}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:sqsd"
    name = "ErrorVisibilityTimeout"
    value = "0"
  }

  # Vpc  and network related settings
  setting {
    namespace = "aws:ec2:vpc"
    name      = "VPCId"
    value     = "${var.vpc_id}"
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "Subnets"
    value     = "${var.ec2_subnet_ids}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "MonitoringInterval"
    value     = "${var.auto_scaling_lc_monitoring_interval}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "SecurityGroups"
    value     = "${var.auto_scaling_lc_security_groups}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "APP"
    value     = "${var.eb_application_name}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "STATSD_HOST_PORT"
    value     = "${var.statsd_host}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "ENVIRONMENT"
    value     = "${var.env}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "S3_BUCKET"
    value     = "${var.s3_bucket}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "METADATA_TABLE"
    value     = "${var.metadata_table}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "PUBSUB_ENDPOINT"
    value     = "${var.pubsub_endpoint}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "TOPIC_PREFIX"
    value     = "${var.topic_prefix}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "LOG_LEVEL"
    value     = "${var.log_level}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:environment"
    name      = "ServiceRole"
    value     = "${var.eb_environment_service_role}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name = "DeploymentPolicy"
    value = "Rolling"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name = "BatchSize"
    value = "34"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name = "BatchSizeType"
    value = "Percentage"
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name = "RollingUpdateType"
    value = "Health"
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name = "MaxBatchSize"
    value = "2"
  }
  
  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name = "MinInstancesInService"
    value = "2"
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name = "SystemType"
    value = "enhanced"
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name = "ConfigDocument"
    value = "{\"Version\":1,\"CloudWatchMetrics\":{\"Instance\":{\"RootFilesystemUtil\":60,\"CPUIrq\":60,\"LoadAverage5min\":60,\"ApplicationRequests5xx\":60,\"ApplicationRequests4xx\":60,\"CPUUser\":60,\"LoadAverage1min\":60,\"ApplicationLatencyP50\":60,\"CPUIdle\":60,\"InstanceHealth\":60,\"ApplicationLatencyP95\":60,\"ApplicationLatencyP85\":60,\"ApplicationLatencyP90\":60,\"CPUSystem\":60,\"ApplicationLatencyP75\":60,\"CPUSoftirq\":60,\"ApplicationLatencyP10\":60,\"ApplicationLatencyP99\":60,\"ApplicationRequestsTotal\":60,\"ApplicationLatencyP99.9\":60,\"ApplicationRequests3xx\":60,\"ApplicationRequests2xx\":60,\"CPUIowait\":60,\"CPUNice\":60},\"Environment\":{\"InstancesSevere\":60,\"InstancesDegraded\":60,\"ApplicationRequests5xx\":60,\"ApplicationRequests4xx\":60,\"ApplicationLatencyP50\":60,\"ApplicationLatencyP95\":60,\"ApplicationLatencyP85\":60,\"InstancesUnknown\":60,\"ApplicationLatencyP90\":60,\"InstancesInfo\":60,\"InstancesPending\":60,\"ApplicationLatencyP75\":60,\"ApplicationLatencyP10\":60,\"ApplicationLatencyP99\":60,\"ApplicationRequestsTotal\":60,\"InstancesNoData\":60,\"ApplicationLatencyP99.9\":60,\"ApplicationRequests3xx\":60,\"ApplicationRequests2xx\":60,\"InstancesOk\":60,\"InstancesWarning\":60}}}"
  }

  setting {
    namespace = "aws:elasticbeanstalk:cloudwatch:logs"
    name = "StreamLogs"
    value = "true"
  }
}

resource "aws_sns_topic" "callback" {
  name = "${var.common_name}-callback"
}

resource "aws_sns_topic_subscription" "callback" {
  topic_arn = "${aws_sns_topic.callback.arn}"
  protocol = "sqs"
  endpoint = "${var.worker_queue_arn}"
}
