package cwrapper

import (
	"os"
	"path/filepath"
	"testing"

	"code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/upload-service/transformations"
	"github.com/stretchr/testify/require"
	"gopkg.in/gographics/imagick.v2/imagick"
)

var imagesToTest []string

func init() {
	root := "../../test_images/"
	filepath.Walk(root, func(path string, info os.FileInfo, err error) error {
		if err != nil {
			return err
		}
		if path != root {
			imagesToTest = append(imagesToTest, path)
		}
		return nil
	})
}

func runTest(t *testing.T, path string) {
	mw := imagick.NewMagickWand()
	defer mw.Destroy()
	req := require.New(t)

	resize := &transformations.ResizeDimensions{
		Width:  1024,
		Height: 768,
	}
	transcode := &transformations.Transcode{
		Format:      "jpeg",
		RemoveAlpha: uploader.RemoveAlpha_TRUE,
	}

	err := mw.ReadImage(path)
	req.NoError(err)

	err = ApplyTransformation(resize, mw)
	req.NoError(err, "error applying resize", err)

	err = ApplyTransformation(transcode, mw)
	req.NoError(err, "error applying transcode", err)

	req.Equal(mw.GetImageFormat(), "jpeg")

	// NOTE: Leave this commented code here, it's convenient to manually inspect the quality of these images.
	// Play with the transformations that are applied and look at the results :)
	//out := filepath.Dir(path) + "/mod-" + filepath.Base(path) + ".jpg"
	//err = mw.WriteImage(out)
	//req.Nil(err, "error writing image", err)
}

func TestSmoke(t *testing.T) {
	require.NotEmpty(t, imagesToTest)
	imagick.Initialize()
	defer imagick.Terminate()

	for _, path := range imagesToTest {
		t.Run(filepath.Base(path), func(t *testing.T) { runTest(t, path) })
	}
}
