package cwrapper

import (
	"testing"

	"gopkg.in/gographics/imagick.v2/imagick"

	"fmt"

	"code.justin.tv/web/upload-service/transformations"
	"github.com/stretchr/testify/suite"
)

type ResizeTestSuite struct {
	suite.Suite
	mw    *imagick.MagickWand
	ratio float64
}

func (suite *ResizeTestSuite) SetupSuite() {
	imagick.Initialize()
}

func (suite *ResizeTestSuite) SetupTest() {
	suite.mw = imagick.NewMagickWand()

	err := suite.mw.ReadImage("../../test_images/me.jpg")
	if err != nil {
		fmt.Println("error reading", err.Error(), err)
	}

	suite.ratio = float64(suite.mw.GetImageWidth()) / float64(suite.mw.GetImageHeight())
}

func (suite *ResizeTestSuite) TearDownTest() {
	suite.mw.Destroy()
}

func (suite *ResizeTestSuite) TearDownSuite() {
	imagick.Terminate()
}

func (suite *ResizeTestSuite) TestResizeDimensionsBoth() {
	c := transformations.ResizeDimensions{Height: 10, Width: 15}
	err := ApplyTransformation(&c, suite.mw)
	suite.Require().NoError(err)

	suite.Require().Equal(suite.mw.GetImageWidth(), uint(15))
	suite.Require().Equal(suite.mw.GetImageHeight(), uint(10))
}

func (suite *ResizeTestSuite) TestResizeDimensionsWidth() {
	c := transformations.ResizeDimensions{Width: 10}
	err := ApplyTransformation(&c, suite.mw)
	suite.Require().NoError(err)

	suite.Require().Equal(suite.mw.GetImageWidth(), uint(10))
	suite.Require().InEpsilon(float64(suite.mw.GetImageHeight()), suite.ratio*10.0, 1.0)
}

func (suite *ResizeTestSuite) TestResizeDimensionsHeight() {
	c := transformations.ResizeDimensions{Height: 10}
	err := ApplyTransformation(&c, suite.mw)
	suite.Require().NoError(err)

	suite.Require().InEpsilon(float64(suite.mw.GetImageWidth()), 10.0/suite.ratio, 1.0)
	suite.Require().Equal(suite.mw.GetImageHeight(), uint(10))
}

func (suite *ResizeTestSuite) TestResizeDimensionsPercentage() {
	width := float64(suite.mw.GetImageWidth())
	height := float64(suite.mw.GetImageHeight())

	c := transformations.ResizePercentage{Percent: 75}
	err := ApplyTransformation(&c, suite.mw)
	suite.Require().NoError(err)

	suite.Require().Equal(suite.mw.GetImageWidth(), uint(width*.75))
	suite.Require().Equal(suite.mw.GetImageHeight(), uint(height*.75))
}

func TestResize(t *testing.T) {
	suite.Run(t, new(ResizeTestSuite))
}
