package cwrapper

import (
	"fmt"
	"os"
	"testing"

	"code.justin.tv/web/upload-service/transformations"
	"github.com/stretchr/testify/suite"
	"gopkg.in/gographics/imagick.v2/imagick"
)

type StripTestSuite struct {
	suite.Suite
}

func (suite *StripTestSuite) SetupSuite() {
	imagick.Initialize()
}

func (suite *StripTestSuite) TearDownSuite() {
	imagick.Terminate()
}

func (suite *StripTestSuite) TestStripImageWithNoExif() {
	img := "../../test_images/bread.JPG"
	after := "../../test_images/bread_stripped.JPG"
	mw := imagick.NewMagickWand()
	err := mw.ReadImage(img)
	if err != nil {
		fmt.Println("error reading", err.Error(), err)
	}
	c := transformations.Strip{}
	err = ApplyTransformation(&c, mw)
	suite.Require().NoError(err)
	preExif := mw.IdentifyImage()

	// no exif here
	suite.Require().NotContains(preExif, "exif")
	mw.Destroy()

	mw = imagick.NewMagickWand()
	mw.ReadImage(after)
	postExif := mw.IdentifyImage()

	// still no exif
	suite.Require().NotContains(postExif, "exif")

	err = os.Remove(after)
	if err != nil {
		fmt.Println("failed to remove stripped exif img", err.Error(), err)
	}
	mw.Destroy()
}

func (suite *StripTestSuite) TestExifDataRemoved() {
	imgs := map[string]string{
		"../../test_images/bread2.JPG":  "../../test_images/bread2_stripped.JPG",
		"../../test_images/bread3.JPG":  "../../test_images/bread3_stripped.JPG",
		"../../test_images/pinball.JPG": "../../test_images/pinball_stripped.JPG",
	}

	for before, after := range imgs {
		mw := imagick.NewMagickWand()
		err := mw.ReadImage(before)
		if err != nil {
			fmt.Println("error reading", err.Error(), err)
		}
		c := transformations.Strip{}
		err = ApplyTransformation(&c, mw)
		suite.Require().NoError(err)
		preExif := mw.IdentifyImage()
		suite.Require().Contains(preExif, "exif")
		mw.WriteImage(after)
		mw.Destroy()

		// test the new image to ensure it has no exif data
		mw = imagick.NewMagickWand()
		mw.ReadImage(after)
		postExif := mw.IdentifyImage()
		suite.Require().NotContains(postExif, "exif")

		err = os.Remove(after)
		if err != nil {
			fmt.Println("failed to remove stripped exif img", err.Error(), err)
		}
		mw.Destroy()
	}
}

func (suite *StripTestSuite) TestCompressionSideEffect() {
	beforeImg := "../../test_images/exif_parrot.JPG"
	afterImg := "../../test_images/exif_parrot_stripped.jpg"
	mw := imagick.NewMagickWand()
	err := mw.ReadImage(beforeImg)
	if err != nil {
		fmt.Println("error reading", err.Error(), err)
	}
	c := transformations.Strip{}
	err = ApplyTransformation(&c, mw)
	suite.Require().NoError(err)
	mw.WriteImage(afterImg)
	suite.Require().True(ensureImgCompression(beforeImg, afterImg))
	err = os.Remove(afterImg)
	if err != nil {
		fmt.Println("failed to remove stripped exif img", err.Error(), err)
	}
	mw.Destroy()
}

func (suite *StripTestSuite) TestOrientationUnchanged() {
	imgs := []string{
		"../../test_images/bread.JPG",
		"../../test_images/bread2.JPG",
		"../../test_images/bread3.JPG",
		"../../test_images/pinball.JPG",
	}

	for _, img := range imgs {
		mw := imagick.NewMagickWand()
		err := mw.ReadImage(img)
		if err != nil {
			fmt.Println("error reading", err.Error(), err)
		}
		c := transformations.Strip{}
		err = ApplyTransformation(&c, mw)
		suite.Require().NoError(err)

		// the orientation must be 1 (top-left) in order for the strip
		// to not have affected the orientation of the image
		suite.Require().Equal(mw.GetImageOrientation(), imagick.OrientationType(1))
		mw.Destroy()
	}
}

func ensureImgCompression(img1, img2 string) bool {
	imgFile1, err := os.Open(img1)
	if err != nil {
		fmt.Println("error opening file", err.Error(), err)
	}
	defer imgFile1.Close()

	imgFile2, err := os.Open(img2)
	if err != nil {
		fmt.Println("error opening file", err.Error(), err)
	}
	defer imgFile2.Close()

	sizeImg1, err := imgFile1.Stat()
	if err != nil {
		fmt.Println("error getting img1 size", err.Error(), err)
	}

	sizeImg2, err := imgFile2.Stat()
	if err != nil {
		fmt.Println("error getting img2 size", err.Error(), err)
	}
	return sizeImg1.Size() > sizeImg2.Size()
}

func TestStrip(t *testing.T) {
	suite.Run(t, new(StripTestSuite))
}
