package cwrapper

import (
	"errors"
	"testing"

	"gopkg.in/gographics/imagick.v2/imagick"

	"fmt"

	"code.justin.tv/web/upload-service/rpc/uploader"
	"code.justin.tv/web/upload-service/transformations"
	"github.com/stretchr/testify/suite"
)

type TranscodeTestSuite struct {
	suite.Suite
	mw *imagick.MagickWand
}

func (suite *TranscodeTestSuite) SetupSuite() {
	imagick.Initialize()
}

func (suite *TranscodeTestSuite) SetupTest() {
	suite.mw = imagick.NewMagickWand()

	err := suite.mw.ReadImage("../../test_images/horse3.png")
	if err != nil {
		fmt.Println("error reading", err.Error(), err)
	}

	hasAlpha := suite.mw.GetImageAlphaChannel()
	suite.True(hasAlpha, "these tests require an input image with an alpha channel")
}

func (suite *TranscodeTestSuite) TearDownTest() {
	suite.mw.Destroy()
}

func (suite *TranscodeTestSuite) TearDownSuite() {
	imagick.Terminate()
}

// suite.mw.GetImageAlphaChannel will still return true even after
// suite.mw.SetImageAlphaChannel(imagick.ALPHA_CHANNEL_REMOVE) has been called. By looking at the color of the pixel,
// we can see if it has 4 channels or 3.
func (suite *TranscodeTestSuite) RequireAlphaChannel(hasAlpha bool) {
	it := suite.mw.NewPixelRegionIterator(50, 50, 1, 1)
	pixels := it.GetNextIteratorRow()
	color := pixels[0].GetColorAsString()
	if hasAlpha {
		suite.Require().Regexp("^srgba\\([0-9]*,[0-9]*,[0-9]*,[0-9]*\\)", color)
	} else {
		suite.Require().Regexp("^srgb\\([0-9]*,[0-9]*,[0-9]*\\)", color)
	}
}

func (suite *TranscodeTestSuite) TestTranscodePNG() {
	c := transformations.Transcode{Format: "png"}
	err := ApplyTransformation(&c, suite.mw)
	suite.Require().NoError(err)

	suite.Require().Equal(suite.mw.GetImageFormat(), "png")
	suite.RequireAlphaChannel(true)
}

func (suite *TranscodeTestSuite) TestTranscodePNGRemoveAlpha() {
	c := transformations.Transcode{Format: "png", RemoveAlpha: uploader.RemoveAlpha_TRUE}
	err := ApplyTransformation(&c, suite.mw)
	suite.Require().NoError(err)

	suite.Require().Equal(suite.mw.GetImageFormat(), "png")
	suite.RequireAlphaChannel(false)
}

func (suite *TranscodeTestSuite) TestTranscodeJPEGDefaultAlpha() {
	c := transformations.Transcode{Format: "jpeg", Quality: uint(70)}
	err := ApplyTransformation(&c, suite.mw)
	suite.Require().NoError(err)

	suite.Require().Equal(suite.mw.GetImageFormat(), "jpeg")
	suite.Require().Equal(suite.mw.GetImageCompressionQuality(), uint(70))
	suite.RequireAlphaChannel(false)
}

func (suite *TranscodeTestSuite) TestTranscodeJPEGRemoveAlphaFalse() {
	c := transformations.Transcode{Format: "jpeg", Quality: uint(70), RemoveAlpha: uploader.RemoveAlpha_FALSE}
	err := ApplyTransformation(&c, suite.mw)
	suite.Require().NoError(err)

	suite.Require().Equal(suite.mw.GetImageFormat(), "jpeg")
	suite.Require().Equal(suite.mw.GetImageCompressionQuality(), uint(70))
	suite.RequireAlphaChannel(true)
}

func (suite *TranscodeTestSuite) TestTranscodeFailure() {
	c := transformations.Transcode{Format: "asdf"}
	err := ApplyTransformation(&c, suite.mw)
	suite.Require().Equal(errors.New("Failed to set image format: 'asdf'"), err)
}

func TestTranscode(t *testing.T) {
	suite.Run(t, new(TranscodeTestSuite))
}
