package cwrapper

import (
	"fmt"

	"github.com/c2h5oh/datasize"
	"github.com/pkg/errors"

	"io"
	"os"
	"path/filepath"

	"strconv"

	"code.justin.tv/common/config"
	"code.justin.tv/web/upload-service/transformations"
	log "github.com/sirupsen/logrus"
	"gopkg.in/gographics/imagick.v2/imagick"
)

const (
	timeoutKey            = "SQSD_TIMEOUT"
	defaultTimeoutSeconds = 180

	// We want to leave time for cleanup tasks like sending SNS and pubsub notifications.
	timeoutGracePeriodSeconds = 10
)

var (
	maxFiles          = -1
	maxImageArea      = -1
	maxThreads        = -1
	maxDiskUsage      = 1 * datasize.GB.Bytes()
	maxCacheMemory    = 1 * datasize.GB.Bytes()
	maxCacheMemoryMap = 1 * datasize.GB.Bytes()
)

func init() {
	config.Register(map[string]string{
		timeoutKey: strconv.Itoa(defaultTimeoutSeconds),
	})
}

type IMagickTransformer struct{}

func (imt *IMagickTransformer) Initialize() {
	imagick.Initialize()
}

func (imt *IMagickTransformer) Terminate() {
	imagick.Terminate()
}

// Runs transformations and saves the output as $TMPDIR/<<uploadID>>/<<idx>>, where idx is the index of this Output
// in the list.
// Returns an open file pointing to the transformed image.
func (imt *IMagickTransformer) RunTransformations(filename, tmpDir, idx string, info *transformations.FileInfo, transformations []transformations.Transformation) (*os.File, error) {
	mw := imagick.NewMagickWand()
	defer mw.Destroy()

	limitResources(mw)

	err := mw.ReadImage(filename)
	if err != nil {
		return nil, errors.Wrapf(err, "Error reading file")
	}
	for i, transformation := range transformations {
		err = ApplyTransformation(transformation, mw)
		if err != nil {
			return nil, errors.Wrapf(err, "Error applying transformation %d", i)
		}
	}

	result, err := os.Create(filepath.Join(tmpDir, idx))
	if err != nil {
		return nil, errors.Wrap(err, "Error creating temp file")
	}
	err = mw.WriteImageFile(result)
	if err != nil {
		return nil, errors.Wrap(err, "Error writing temp file")
	}
	result.Seek(0, io.SeekStart)

	FillInfoFromWand(mw, info)

	return result, nil
}

func limitResources(mw *imagick.MagickWand) {
	timeout, err := strconv.Atoi(config.Resolve(timeoutKey))
	if err != nil {
		log.WithError(err).Warn("Error reading sqsd timeout from config: %s. Defaulting to %d", config.Resolve(timeoutKey), defaultTimeoutSeconds)
		timeout = defaultTimeoutSeconds
	}

	mw.SetResourceLimit(imagick.RESOURCE_TIME, int64(timeout-timeoutGracePeriodSeconds))
	mw.SetResourceLimit(imagick.RESOURCE_FILE, int64(maxFiles))
	mw.SetResourceLimit(imagick.RESOURCE_AREA, int64(maxImageArea))
	mw.SetResourceLimit(imagick.RESOURCE_THREAD, int64(maxThreads))
	mw.SetResourceLimit(imagick.RESOURCE_DISK, int64(maxDiskUsage))
	mw.SetResourceLimit(imagick.RESOURCE_MEMORY, int64(maxCacheMemory))
	mw.SetResourceLimit(imagick.RESOURCE_MAP, int64(maxCacheMemoryMap))
}

func ApplyTransformation(transform transformations.Transformation, mw *imagick.MagickWand) error {
	if t, ok := transform.(*transformations.AspectRatio); ok {
		return AspectRatio(t, mw)
	} else if t, ok := transform.(*transformations.Crop); ok {
		return Crop(t, mw)
	} else if t, ok := transform.(*transformations.MaxHeight); ok {
		return MaxHeight(t, mw)
	} else if t, ok := transform.(*transformations.MaxWidth); ok {
		return MaxWidth(t, mw)
	} else if t, ok := transform.(*transformations.ResizeDimensions); ok {
		return ResizeDimensions(t, mw)
	} else if t, ok := transform.(*transformations.ResizePercentage); ok {
		return ResizePercentage(t, mw)
	} else if t, ok := transform.(*transformations.Transcode); ok {
		return Transcode(t, mw)
	} else if _, ok := transform.(*transformations.Strip); ok {
		return Strip(mw)
	} else {
		return fmt.Errorf("Unknown transformation %#v", transform)
	}
}

func FillInfoFromWand(mw *imagick.MagickWand, f *transformations.FileInfo) {
	f.IsImage = true
	f.SetFormat(mw.GetImageFormat())
	f.SetWidth(mw.GetImageWidth())
	f.SetHeight(mw.GetImageHeight())
}

func (imt *IMagickTransformer) FillInfoFromFile(tmpfile *os.File, info *transformations.FileInfo) error {
	// info must already be filled in, we can skip the rest
	if info.IsImage == true {
		return nil
	}

	mw := imagick.NewMagickWand()
	defer mw.Destroy()
	err := mw.PingImage(tmpfile.Name())
	if err != nil {
		return err
	}

	stats, err := tmpfile.Stat()
	if err != nil {
		return errors.Wrap(err, "Error statting file on disk")
	}
	info.Size = stats.Size()

	FillInfoFromWand(mw, info)
	return nil
}
