package transformations

import (
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestImageFormat(t *testing.T) {
	formats := []struct {
		imagick  string
		expected string
	}{
		{"", ""},
		{"jpg", "jpg"},
		{"jpeg", "jpg"},
		{"bmp", "bmp"},
		{"bmp2000", "bmp"},
		{"2000bmp2000", "2000bmp2000"},
		{"png", "png"},
		{"png2000", "png"},
		{"2000png2000", "2000png2000"},
		{"gif", "gif"},
		{"gif2000", "gif"},
		{"2000gif2000", "2000gif2000"},
		{"pdf", "pdf"},
		{"doc", "doc"},
		{"mp4", "mp4"},
		{"avi", "avi"},
		{"unknown", "unknown"},
	}

	for _, format := range formats {
		info := NewFileInfo("upload-id")
		info.SetFormat(format.imagick)
		assert.Equal(t, format.expected, info.Format)
	}
}

func TestReplaceParams(t *testing.T) {
	info := NewFileInfo("upload-id")
	info.IsImage = true
	info.Size = int64(13016)
	info.SetWidth(195)
	info.SetHeight(209)
	info.SetFormat("jpg")

	result := info.ExecTemplate("asdf{{UploadID}}_{{UploadID}}__{{Dimensions}}{{Extension}}")
	require.Equal(t, "asdfupload-id_upload-id__195x209.jpg", result)
}

func TestReplaceParamsEmptyInfo(t *testing.T) {
	info := NewFileInfo("upload-id")

	result := info.ExecTemplate("asdf{{UploadID}}_{{UploadID}}__{{Dimensions}}{{Extension}}")
	require.Equal(t, "asdfupload-id_upload-id__", result)
}

func TestMimeType(t *testing.T) {
	formatMimeTypes := []struct {
		format   string
		mimeType string
	}{
		{"", ""},
		{"invalid", ""},
		{"jpg", "image/jpeg"},
		{"png", "image/png"},
		{"gif", "image/gif"},
	}

	for _, formatMimeType := range formatMimeTypes {
		info := NewFileInfo("upload-id")
		info.SetFormat(formatMimeType.format)
		mimeType := info.MimeType()
		require.Equal(t, formatMimeType.mimeType, mimeType)
	}
}
