package client

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"strconv"

	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"

	"code.justin.tv/common/twitchhttp"
)

const (
	defaultStatSampleRate = 1.0

	createReportPath        = "/reports"
	tosSuspensionReportPath = "/suspensions"

	xactName = "leviathan"

	maxBodySlurpSize = int64(2 << 10)
)

// CreateReportParams contains the information necessary to create a report.
type CreateReportParams struct {
	FromUserID   string
	TargetUserID string
	Reason       string
	Description  string
	Content      string
	Origin       string
	ContentID    string
	Extra        string
}

type TosSuspensionReportParam struct {
	FromUserID     string
	TargetUserID   string
	Reason         string
	DetailedReason string
	Description    string
	Content        string
	Origin         string
	Duration       int64
	IPBan          int64
	ClearImages    bool
}

// Client facilitates sending messages to the Leviathan service.
type Client interface {
	CreateReport(ctx context.Context, createReportParams CreateReportParams) error
	TosSuspensionReport(ctx context.Context, tosSuspensionReport TosSuspensionReportParam) error
}

// Config contains the configuration necessary to create a Leviathan client.
type Config struct {
	Host               string
	AuthorizationToken string
	TransportConf      twitchhttp.TransportConf
}

// New creates a Leviathan client.
func New(conf Config, stats statsd.Statter) (Client, error) {
	twitchClient, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host:           conf.Host,
		Transport:      conf.TransportConf,
		Stats:          stats,
		TimingXactName: xactName,
	})
	if err != nil {
		return nil, err
	}

	return &leviathanImpl{
		Client:             twitchClient,
		authorizationToken: conf.AuthorizationToken,
	}, nil
}

type createReportRequestBody struct {
	AuthorizationToken string       `json:"authorization_token"`
	ReportParams       reportParams `json:"report"`
}

type tosSuspensionReportRequestBody struct {
	AuthorizationToken string                    `json:"authorization_token"`
	ReportParams       tosSuspensionReportParams `json:"suspension"`
}

type reportParams struct {
	FromUserID   int64  `json:"from_user_id"`
	TargetUserID int64  `json:"target_user_id"`
	Reason       string `json:"reason"`
	Description  string `json:"description"`
	Content      string `json:"content"`
	Origin       string `json:"origin,omitempty"`
	ContentID    string `json:"content_id,omitempty"`
	Extra        string `json:"extra1,omitempty"`
}

type tosSuspensionReportParams struct {
	FromUserID     string  `json:"from_user_id"`
	TargetUserID   int64  `json:"target_user_id"`
	Reason         string `json:"reason"`
	Description    string `json:"description"`
	Content        string `json:"content,omitempty"`
	Origin         string `json:"origin"`
	Duration       int64  `json:"duration"`
	IPBan          int64  `json:"ip_ban"`
	ClearImages    bool   `json:"cleared_channel_images"`
	DetailedReason string `json:"detailed_reason,omitempty"`
}

type leviathanImpl struct {
	twitchhttp.Client
	authorizationToken string
}

// Enforce that leviathanImpl implements Client.
var _ Client = leviathanImpl{}

func (d leviathanImpl) CreateReport(ctx context.Context, createReportParams CreateReportParams) (retErr error) {

	reportParams, err := d.parseCreateReportParams(createReportParams)
	if err != nil {
		return err
	}

	requestBody := createReportRequestBody{
		AuthorizationToken: d.authorizationToken,
		ReportParams:       *reportParams,
	}

	body, err := json.Marshal(requestBody)
	if err != nil {
		return err
	}

	req, err := d.NewRequest("POST", createReportPath, bytes.NewBuffer(body))
	if err != nil {
		return err
	}
	req.Header.Add("Content-Type", "application/json")

	resp, err := d.Do(ctx, req, twitchhttp.ReqOpts{
		StatName:       "service.leviathan.create_report",
		StatSampleRate: defaultStatSampleRate,
	})
	if err != nil {
		return err
	}
	defer cleanupResponseBody(resp.Body, &retErr)

	if resp.StatusCode != http.StatusOK && resp.StatusCode != http.StatusCreated {
		errResp := &errResponse{
			code: resp.StatusCode,
		}

		_, err2 := io.CopyN(&errResp.body, resp.Body, maxBodySlurpSize)
		if err2 != nil && err2 != io.EOF {
			return err2
		}

		return errResp
	}

	return nil
}

func (d leviathanImpl) parseCreateReportParams(params CreateReportParams) (*reportParams, error) {
	targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64)
	if err != nil {
		return nil, err
	}

	fromUserID, err := strconv.ParseInt(params.FromUserID, 10, 64)
	if err != nil {
		return nil, err
	}

	return &reportParams{
		FromUserID:   fromUserID,
		TargetUserID: targetUserID,
		Reason:       params.Reason,
		Description:  params.Description,
		Content:      params.Content,
		Origin:       params.Origin,
		ContentID:    params.ContentID,
		Extra:        params.Extra,
	}, nil
}

func (d leviathanImpl) TosSuspensionReport(ctx context.Context, tosSuspensionReport TosSuspensionReportParam) (retErr error) {

	tosSuspensionReportParams, err := d.parseTosSuspensionReportParams(tosSuspensionReport)
	if err != nil {
		return err
	}

	requestBody := tosSuspensionReportRequestBody{
		AuthorizationToken: d.authorizationToken,
		ReportParams:       *tosSuspensionReportParams,
	}

	body, err := json.Marshal(requestBody)
	if err != nil {
		return err
	}

	req, err := d.NewRequest("POST", tosSuspensionReportPath, bytes.NewBuffer(body))
	if err != nil {
		return err
	}
	req.Header.Add("Content-Type", "application/json")

	resp, err := d.Do(ctx, req, twitchhttp.ReqOpts{
		StatName:       "service.leviathan.tos_suspension_report",
		StatSampleRate: defaultStatSampleRate,
	})
	if err != nil {
		return err
	}
	defer cleanupResponseBody(resp.Body, &retErr)

	if resp.StatusCode != http.StatusOK && resp.StatusCode != http.StatusCreated {
		errResp := &errResponse{
			code: resp.StatusCode,
		}

		_, err2 := io.CopyN(&errResp.body, resp.Body, maxBodySlurpSize)
		if err2 != nil && err2 != io.EOF {
			return err2
		}

		return errResp
	}

	return nil
}

func (d leviathanImpl) parseTosSuspensionReportParams(params TosSuspensionReportParam) (*tosSuspensionReportParams, error) {
	targetUserID, err := strconv.ParseInt(params.TargetUserID, 10, 64)
	if err != nil {
		return nil, err
	}

	return &tosSuspensionReportParams{
		FromUserID:     params.FromUserID,
		TargetUserID:   targetUserID,
		Reason:         params.Reason,
		Description:    params.Description,
		Content:        params.Content,
		Origin:         params.Origin,
		DetailedReason: params.DetailedReason,
		Duration:       params.Duration,
		IPBan:          params.IPBan,
		ClearImages:    params.ClearImages,
	}, nil
}

func cleanupResponseBody(body io.ReadCloser, retErr *error) {
	// Read the body so the underlying TCP connection will be re-used.
	_, err := io.CopyN(ioutil.Discard, body, maxBodySlurpSize)
	if err != io.EOF && err != nil && *retErr == nil {
		*retErr = err
	}

	err = body.Close()
	if err != nil && *retErr == nil {
		*retErr = err
	}
}

type errResponse struct {
	code int
	body bytes.Buffer
}

func (e *errResponse) Error() string {
	return fmt.Sprintf("invalid status code: %d: %s", e.code, e.body.String())
}

func (e *errResponse) HTTPCode() int {
	return e.code
}

// NoopClient is an implementation of Client that does nothing.
type NoopClient struct{}

// Enforce that NoopClient implements Client.
var _ Client = NoopClient{}

func (n NoopClient) CreateReport(ctx context.Context, reportParams CreateReportParams) error {
	return nil
}

func (n NoopClient) TosSuspensionReport(ctx context.Context, tosSuspensionReport TosSuspensionReportParam) error {
	return nil
}
