package history

import (
	"bytes"
	"context"
	"encoding/json"
	"log"
	"net/http"
	"strconv"
	"time"

	"github.com/google/go-querystring/query"
	uuid "github.com/satori/go.uuid"

	"code.justin.tv/common/twitchhttp"
)

const (
	addStatName           = "service.history_service.add"
	defaultStatSampleRate = 1.0
	defaultTimingXactName = "history"
	maxRetries            = 5
	region                = "us-west-2"
	searchStatName        = "service.history_service.search"
	// this is a v4 uuid used to generate a v5 uuid.
	// this string is shared with the history-client-ruby repo
	uuidNamespaceStr = "d45db28b-34ed-4738-8f7a-db2a993feadb"
)

// Audit is a struct representing an Audit. Note that when adding audits, `Uuid`
// is optional and `ExpiredAt` should not be passed in as it is calculated
// internally based on the `Expiry` field
type Audit struct {
	UUID         string      `json:"uuid"`
	Action       string      `json:"action"`
	UserType     string      `json:"user_type"`
	UserID       string      `json:"user_id"`
	ResourceType string      `json:"resource_type"`
	ResourceID   string      `json:"resource_id"`
	Description  string      `json:"description"`
	CreatedAt    time.Time   `json:"created_at"`
	ExpiredAt    time.Time   `json:"expired_at,omitempty"`
	Expiry       int64       `json:"expiry"`
	Changes      []ChangeSet `json:"changes"`
}

// ChangeSet is a struct representing an attribute changing from an old value to
// a new value
type ChangeSet struct {
	Attribute string `json:"attribute"`
	OldValue  string `json:"old_value"`
	NewValue  string `json:"new_value"`
}

// SearchParams is a struct representing the params available to the search API
type SearchParams struct {
	Action       string    `url:"action,omitempty" json:"action"`
	UserType     string    `url:"user_type,omitempty" json:"user_type"`
	UserID       string    `url:"user_id,omitempty" json:"user_id"`
	ResourceType string    `url:"resource_type,omitempty" json:"resource_type"`
	ResourceID   string    `url:"resource_id,omitempty" json:"resource_id"`
	Attribute    string    `url:"attribute,omitempty" json:"attribute"`
	OldValue     string    `url:"old_value,omitempty" json:"old_value"`
	NewValue     string    `url:"new_value,omitempty" json:"new_value"`
	CreatedAtLt  time.Time `url:"created_at_lt,omitempty" json:"created_at_lt,omitempty"`
	CreatedAtLte time.Time `url:"created_at_lte,omitempty" json:"created_at_lte,omitempty"`
	CreatedAtGt  time.Time `url:"created_at_gt,omitempty" json:"created_at_gt,omitempty"`
	CreatedAtGte time.Time `url:"created_at_gte,omitempty" json:"created_at_gte,omitempty"`
	Page         int64     `url:"page,omitempty" json:"page,omitempty"`
	PerPage      int64     `url:"per_page,omitempty" json:"per_page,omitempty"`
}

// SearchResults is a struct representing the response from the search API
type SearchResults struct {
	Audits     []*Audit `json:"audits"`
	TotalPages int64    `json:"total_pages"`
}

// Client is the interface this client is expected to implement
type Client interface {
	Add(ctx context.Context, audit Audit) error
	Search(ctx context.Context, params *SearchParams) (*SearchResults, error)
}

type clientImpl struct {
	twitchhttp.Client
}

// InvalidUUIDError is the error returned for an invalid UUID
type InvalidUUIDError struct{}

func (e *InvalidUUIDError) Error() string {
	return "Invalid UUID"
}

// NewClient creates a new client for history service
func NewClient(conf twitchhttp.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}

	twitchClient, err := twitchhttp.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return &clientImpl{twitchClient}, nil
}

// GenUUIDName generates a name for generating a v5 UUID by concatenating
// the values of an audit to a single string
func GenUUIDName(audit Audit) string {
	var buffer bytes.Buffer

	buffer.WriteString(audit.Action)
	buffer.WriteString(audit.UserType)
	buffer.WriteString(audit.UserID)
	buffer.WriteString(audit.ResourceType)
	buffer.WriteString(audit.ResourceID)
	buffer.WriteString(audit.Description)
	buffer.WriteString(audit.CreatedAt.String())
	buffer.WriteString(strconv.FormatInt(audit.Expiry, 10))

	return buffer.String()
}

// Add submits a new audit to history service
func (c *clientImpl) Add(ctx context.Context, audit Audit) error {
	path := "/v1/audits?"

	if audit.UUID == "" {
		uuidName := GenUUIDName(audit)
		uuidNamespace, err := uuid.FromString(uuidNamespaceStr)
		if err != nil {
			return err
		}

		audit.UUID = uuid.NewV5(uuidNamespace, uuidName).String()
	}

	if len(audit.UUID) != 36 {
		return &InvalidUUIDError{}
	}

	bodyJSON, err := json.Marshal(audit)
	if err != nil {
		return err
	}

	req, err := c.NewRequest("POST", path, bytes.NewBuffer(bodyJSON))
	if err != nil {
		return err
	}
	req.Header.Set("Accept", "application/json")
	req.Header.Set("Content-Type", "application/json")

	reqOpts := twitchhttp.ReqOpts{
		StatName:       addStatName,
		StatSampleRate: defaultStatSampleRate,
	}

	var httpResp *http.Response
	for try := 1; try <= maxRetries; try++ {
		httpResp, err = c.Do(ctx, req, reqOpts)
		if err == nil {
			break
		}

		if err != nil && try == maxRetries {
			return err
		}
	}

	defer func() {
		err = httpResp.Body.Close()
		if err != nil {
			log.Printf("unable to close response body: %q", err)
		}
	}()

	switch httpResp.StatusCode {
	case http.StatusOK:
		return nil
	default:
		return twitchhttp.HandleFailedResponse(httpResp)
	}
}

// Search searches for an audit matching the params specified
func (c *clientImpl) Search(ctx context.Context, params *SearchParams) (*SearchResults, error) {
	path := "/v1/audits?"

	v, err := query.Values(params)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("GET", path+v.Encode(), nil)
	if err != nil {
		return nil, err
	}
	req.Header.Set("Accept", "application/json")
	req.Header.Set("Content-Type", "application/json")

	reqOpts := twitchhttp.ReqOpts{
		StatName:       searchStatName,
		StatSampleRate: defaultStatSampleRate,
	}

	httpResp, err := c.Do(ctx, req, reqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		err = httpResp.Body.Close()
	}()

	var results SearchResults

	switch httpResp.StatusCode {
	case http.StatusOK:
		dec := json.NewDecoder(httpResp.Body)
		err = dec.Decode(&results)
		if err != nil {
			return nil, err
		}
	default:
		err = twitchhttp.HandleFailedResponse(httpResp)
	}
	return &results, err
}
