package evs

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/growth/emailvalidator/evs/documents"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "emailvalidator"
)

// Client is the client library to the email validator service
type Client interface {
	// AddVerificationRequest adds a new verification request to the validator service
	// and triggers a "click this link" email to be sent.  It is an error to try
	// to add a verification request for an email that is known to be rejected (on any namespace/key),
	// or to try to re-verify a namespace/key/email triple that is already verified.
	// namespace - namespace of the validation request
	// key - a key that is unique within the given namespace
	// email - the email to be validated
	AddVerificationRequest(ctx context.Context, namespace, key, email, locale, purpose string, reqOpts *twitchclient.ReqOpts) (*documents.VerificationRequestDocument, error)
	VerificationStatus(ctx context.Context, namespace, key, email string, reqOpts *twitchclient.ReqOpts) (*documents.VerificationRequestDocument, error)
	VerificationStatusByOpaqueID(ctx context.Context, opaqueID string, reqOpts *twitchclient.ReqOpts) (*documents.VerificationRequestDocument, error)
	Verify(ctx context.Context, opaqueID string, reqOpts *twitchclient.ReqOpts) (*documents.VerificationRequestDocument, error)
	Reject(ctx context.Context, email string, reqOpts *twitchclient.ReqOpts) error
}

type client struct {
	twitchclient.Client
}

// NewClient allocates a new client
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	return &client{twitchClient}, err
}

func (c *client) AddVerificationRequest(ctx context.Context, namespace, key, email, locale, purpose string, reqOpts *twitchclient.ReqOpts) (*documents.VerificationRequestDocument, error) {
	requestBody := documents.AddVerificationRequest{
		Namespace: namespace,
		Key:       key,
		Email:     email,
		Purpose:   purpose,
		Locale:    locale,
	}

	bodyBytes, err := json.Marshal(requestBody)

	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("POST", "/verification_request", bytes.NewReader(bodyBytes))

	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.emailvalidator.add_verification_request",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.VerificationRequestDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, err
}

func (c *client) VerificationStatus(ctx context.Context, namespace, key, email string, reqOpts *twitchclient.ReqOpts) (*documents.VerificationRequestDocument, error) {
	req, err := c.NewRequest("GET", fmt.Sprintf("/verification_request/%s/%s/%s", namespace, key, email), nil)

	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.emailvalidator.verification_status",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.VerificationRequestDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, err
}

func (c *client) VerificationStatusByOpaqueID(ctx context.Context, opaqueID string, reqOpts *twitchclient.ReqOpts) (*documents.VerificationRequestDocument, error) {
	req, err := c.NewRequest("GET", fmt.Sprintf("/verification_request/%s", opaqueID), nil)

	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.emailvalidator.verification_status_by_opaque_id",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.VerificationRequestDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, err
}

func (c *client) Verify(ctx context.Context, opaqueID string, reqOpts *twitchclient.ReqOpts) (*documents.VerificationRequestDocument, error) {
	req, err := c.NewRequest("POST", fmt.Sprintf("/verify/%s", opaqueID), nil)

	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.emailvalidator.verify",
		StatSampleRate: defaultStatSampleRate,
	})

	var document documents.VerificationRequestDocument

	resp, err := c.DoJSON(ctx, &document, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode != http.StatusOK {
		return nil, twitchclient.HandleFailedResponse(resp)
	}

	return &document, err
}

func (c *client) Reject(ctx context.Context, email string, reqOpts *twitchclient.ReqOpts) error {
	req, err := c.NewRequest("POST", fmt.Sprintf("/reject/%s", email), nil)

	if err != nil {
		return err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.emailvalidator.reject",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		if err2 := resp.Body.Close(); err == nil && err2 != nil {
			err = fmt.Errorf("error closing body: %s", err2)
		}
	}()

	if resp.StatusCode != http.StatusNoContent {
		return twitchclient.HandleFailedResponse(resp)
	}

	return nil
}
