package publisher

import (
	"encoding/json"
	"fmt"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/ec2rolecreds"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/kinesis"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"
)

// Publisher implements the interface for a client publishing events to Kinesis
type Publisher interface {
	Publish(ctx context.Context, e Event) error
}

type publisher struct {
	Client *kinesis.Kinesis
	Stats  statsd.Statter
	Conf   *Config
}

const (
	statsPrefix = "kinesis"
)

// NewPublisher instantiates a new Publisher given a Config and Statsd client
func NewPublisher(c *Config, stats statsd.Statter) (Publisher, error) {
	sess := session.New()
	conf := &aws.Config{Region: aws.String(c.awsRegion())}

	cred := credentials.NewChainCredentials([]credentials.Provider{
		&stscreds.AssumeRoleProvider{
			RoleARN: c.RoleARN,
			Client:  sts.New(sess),
		},
		&ec2rolecreds.EC2RoleProvider{
			Client: ec2metadata.New(sess),
		},
		&credentials.EnvProvider{},
		&credentials.SharedCredentialsProvider{},
	})
	conf.WithCredentials(cred)
	_, err := cred.Get()
	if err != nil {
		return nil, err
	}

	k := kinesis.New(sess, conf)
	return &publisher{Client: k, Stats: stats, Conf: c}, nil
}

func (p *publisher) Publish(ctx context.Context, e Event) error {
	var err error
	for i := 0; i < p.Conf.retryCount(); i++ {
		_, err = p.publish(ctx, e)
		if err == nil {
			break
		}
		time.Sleep(p.Conf.retryDelay())
	}
	if err != nil {
		return err
	}
	return nil
}

func (p *publisher) publish(ctx context.Context, e Event) (*kinesis.PutRecordOutput, error) {
	t0 := time.Now()
	blob, err := json.Marshal(e)
	if err != nil {
		return nil, err
	}
	input := &kinesis.PutRecordInput{
		Data:         blob,
		PartitionKey: aws.String(e.PartitionKey()),
		StreamName:   aws.String(p.Conf.StreamName),
	}
	res, err := p.Client.PutRecord(input)
	dur := time.Since(t0)
	p.reportPublish("put_record", err, dur)
	return res, err
}

func (p *publisher) reportPublish(cmd string, err error, dur time.Duration) {
	_ = p.Stats.TimingDuration(fmt.Sprintf("%s.%s", statsPrefix, cmd), dur, p.Conf.sampleRate())
	var res string
	if err != nil {
		res = "failure"
	} else {
		res = "success"
	}
	_ = p.Stats.Inc(fmt.Sprintf("%s.%s.%s", statsPrefix, cmd, res), 1.0, p.Conf.sampleRate())
}
