package rediser

import (
	"context"
	"fmt"
	"log"
	"sync/atomic"
	"time"

	"github.com/go-redis/redis"

	rc "code.justin.tv/identity/rediser/common"
)

func (r *rediser) Del(ctx context.Context, keys ...string) (int64, error) {
	keys = prefixKeys(keys, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, delCmd, delArgs{keys: keys})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(delArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

// TODO: Accept a fifth argument `statName` which allows a breakdown of EVAL timings
// by specific command. See chat/presence and chat/redis.
func (r *rediser) Eval(ctx context.Context, script string, keys []string, args []interface{}) (interface{}, error) {
	keys = prefixKeys(keys, r.Opts.KeyPrefix)
	val, err := r.cmd(ctx, evalCmd, evalArgs{script: script, keys: keys, args: args})
	if err != nil {
		return nil, err
	}
	return val, nil
}

func (r *rediser) Exists(ctx context.Context, keys ...string) (int64, error) {
	keys = prefixKeys(keys, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, existsCmd, existsArgs{keys: keys, res: 0})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(existsArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) Expire(ctx context.Context, key string, ttl time.Duration) (bool, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, expireCmd, expireArgs{key: key, ttl: ttl})
	if err != nil {
		return false, err
	}
	args, ok := arg.(expireArgs)
	if !ok {
		return false, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) FlushDB(ctx context.Context) error {
	_, err := r.cmd(ctx, flushDbCmd, nil)
	return err
}

func (r *rediser) Get(ctx context.Context, key string) (string, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, getCmd, getArgs{key: key})
	if err != nil {
		return "", err
	}
	args, ok := arg.(getArgs)
	if !ok {
		return "", rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) HDel(ctx context.Context, key string, fields ...string) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, hDelCmd, hDelArgs{key: key, fields: fields})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(hDelArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) HGetAll(ctx context.Context, key string) (map[string]string, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, hGetAllCmd, hGetAllArgs{key: key})
	if err != nil {
		return map[string]string{}, err
	}
	args, ok := arg.(hGetAllArgs)
	if !ok {
		return map[string]string{}, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) HKeys(ctx context.Context, key string) ([]string, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, hKeysCmd, hKeysArgs{key: key})
	if err != nil {
		return []string{}, err
	}
	args, ok := arg.(hKeysArgs)
	if !ok {
		return []string{}, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) HMGet(ctx context.Context, key string, fields ...string) ([]interface{}, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, hMGetCmd, hMGetArgs{key: key, fields: fields})
	if err != nil {
		return []interface{}{}, err
	}
	args, ok := arg.(hMGetArgs)
	if !ok {
		return []interface{}{}, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) HMSet(ctx context.Context, key string, fields map[string]interface{}) error {
	key = prefixKey(key, r.Opts.KeyPrefix)
	_, err := r.cmd(ctx, hMSetCmd, hMSetArgs{key: key, fields: fields})
	return err
}

func (r *rediser) HSet(ctx context.Context, key string, field string, val interface{}) (bool, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, hSetCmd, hSetArgs{key: key, field: field, val: val})
	if err != nil {
		return false, err
	}
	args, ok := arg.(hSetArgs)
	if !ok {
		return false, rc.ErrInvalidArguments
	}
	return args.res, nil

}

func (r *rediser) HSetNX(ctx context.Context, key string, field string, val interface{}) (bool, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, hSetNXCmd, hSetNXArgs{key: key, field: field, val: val})
	if err != nil {
		return false, err
	}
	args, ok := arg.(hSetNXArgs)
	if !ok {
		return false, rc.ErrInvalidArguments
	}
	return args.res, nil

}

func (r *rediser) HIncrBy(ctx context.Context, key string, field string, val int64) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, hIncrByCmd, hIncrByArgs{key: key, field: field, val: val})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(hIncrByArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) HLen(ctx context.Context, key string) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, hLenCmd, hLenArgs{key: key})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(hLenArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) Incr(ctx context.Context, key string) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, incrCmd, incrArgs{key: key})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(incrArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) LLen(ctx context.Context, key string) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, lLenCmd, lLenArgs{key: key})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(lLenArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) LRange(ctx context.Context, key string, start int64, stop int64) ([]string, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, lRangeCmd, lRangeArgs{key: key, start: start, stop: stop})
	if err != nil {
		return []string{}, err
	}
	args, ok := arg.(lRangeArgs)
	if !ok {
		return []string{}, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) LRem(ctx context.Context, key string, count int64, val interface{}) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, lRemCmd, lRemArgs{key: key, count: count, val: val})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(lRemArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) LTrim(ctx context.Context, key string, start, stop int64) error {
	key = prefixKey(key, r.Opts.KeyPrefix)
	if _, err := r.cmd(ctx, lTrimCmd, lTrimArgs{key: key, start: start, stop: stop}); err != nil {
		return err
	}
	return nil
}

func (r *rediser) MGet(ctx context.Context, keys ...string) ([]interface{}, error) {
	keys = prefixKeys(keys, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, mGetCmd, mGetArgs{keys: keys})
	if err != nil {
		return []interface{}{}, err
	}
	args, ok := arg.(mGetArgs)
	if !ok {
		return []interface{}{}, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) MSet(ctx context.Context, pairs ...interface{}) error {
	var err error
	pairs, err = prefixPairs(pairs, r.Opts.KeyPrefix)
	if err != nil {
		return err
	}
	_, err = r.cmd(ctx, mSetCmd, mSetArgs{pairs: pairs})
	return err
}

func (r *rediser) MSetNxWithTTL(ctx context.Context, ttl time.Duration, pairs ...interface{}) error {
	var err error
	pairs, err = prefixPairs(pairs, r.Opts.KeyPrefix)
	if err != nil {
		return err
	}
	_, err = r.cmd(ctx, mSetNxTTLCmd, mSetTTLArgs{pairs: pairs, ttl: ttl})
	if err != nil {
		return err
	}
	return nil
}

func (r *rediser) MSetWithTTL(ctx context.Context, ttl time.Duration, pairs ...interface{}) error {
	var err error
	pairs, err = prefixPairs(pairs, r.Opts.KeyPrefix)
	if err != nil {
		return err
	}
	_, err = r.cmd(ctx, mSetTTLCmd, mSetTTLArgs{pairs: pairs, ttl: ttl})
	return err
}

func (r *rediser) PipelinedGet(ctx context.Context, keys ...string) ([]interface{}, error) {
	keys = prefixKeys(keys, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, pipelinedGetCmd, pipelinedGetArgs{keys: keys})
	if err != nil {
		return nil, err
	}
	args, ok := arg.(pipelinedGetArgs)
	if !ok {
		return nil, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) PipelinedInvalidate(ctx context.Context, keys ...string) (int64, error) {
	keys = prefixKeys(keys, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, pipelinedDelCmd, pipelinedDelArgs{keys: keys})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(pipelinedDelArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) RPush(ctx context.Context, key, val string) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, rPushCmd, rPushArgs{key: key, val: val})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(rPushArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) SAdd(ctx context.Context, key string, vals ...interface{}) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, sAddCmd, sAddArgs{key: key, vals: vals})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(sAddArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) SDiff(ctx context.Context, keys ...string) ([]string, error) {
	keys = prefixKeys(keys, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, sDiffCmd, sDiffArgs{keys: keys})
	if err != nil {
		return []string{}, err
	}
	args, ok := arg.(sDiffArgs)
	if !ok {
		return []string{}, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) Set(ctx context.Context, key, val string, ttl time.Duration) error {
	key = prefixKey(key, r.Opts.KeyPrefix)
	if _, err := r.cmd(ctx, setCmd, setArgs{key: key, val: val, ttl: ttl}); err != nil {
		return err
	}
	return nil
}

func (r *rediser) SetNX(ctx context.Context, key, val string, ttl time.Duration) (bool, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, setNxCmd, setNxArgs{key: key, val: val, ttl: ttl})
	if err != nil {
		return false, err
	}
	args, ok := arg.(setNxArgs)
	if !ok {
		return false, err
	}
	return args.res, nil
}

func (r *rediser) SIsMember(ctx context.Context, key string, val interface{}) (bool, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, sIsMemberCmd, sIsMemberArgs{key: key, val: val})
	if err != nil {
		return false, err
	}
	args, ok := arg.(sIsMemberArgs)
	if !ok {
		return false, err
	}
	return args.res, nil
}

func (r *rediser) SMembers(ctx context.Context, key string) ([]string, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, sMembersCmd, sMembersArgs{key: key})
	if err != nil {
		return nil, err
	}
	args, ok := arg.(sMembersArgs)
	if !ok {
		return nil, err
	}
	return args.res, nil
}

func (r *rediser) SRem(ctx context.Context, key string, vals ...interface{}) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, sRemCmd, sRemArgs{key: key, vals: vals})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(sRemArgs)
	if !ok {
		return 0, err
	}
	return args.res, nil
}

func (r *rediser) SScan(ctx context.Context, key string, cursor uint64, match string, count int64) (uint64, []string, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, sScanCmd, sScanArgs{key: key, cursor: cursor, match: match, count: count})
	if err != nil {
		return 0, nil, err
	}
	args, ok := arg.(sScanArgs)
	if !ok {
		return 0, nil, rc.ErrInvalidArguments
	}
	return args.resCursor, args.res, nil
}

func (r *rediser) TTL(ctx context.Context, key string) (time.Duration, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, ttlCmd, ttlArgs{key: key})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(ttlArgs)
	if !ok {
		return 0, err
	}
	return args.res, nil
}

func (r *rediser) ZAdd(ctx context.Context, key string, items ...redis.Z) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, zAddCmd, zAddArgs{key: key, items: items})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(zAddArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) ZCard(ctx context.Context, key string) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, zCardCmd, zCardArgs{key: key})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(zCardArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) ZRangeByLex(ctx context.Context, key string, opt redis.ZRangeBy) ([]string, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, zRangeByLexCmd, zRangeByLexArgs{key: key, opt: opt})
	if err != nil {
		return []string{}, err
	}
	args, ok := arg.(zRangeByLexArgs)
	if !ok {
		return []string{}, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) ZRangeByScore(ctx context.Context, key string, opt redis.ZRangeBy) ([]string, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, zRangeByScoreCmd, zRangeByScoreArgs{key: key, opt: opt})
	if err != nil {
		return []string{}, err
	}
	args, ok := arg.(zRangeByScoreArgs)
	if !ok {
		return []string{}, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) ZRem(ctx context.Context, key, element string) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, zRemCmd, zRemArgs{key: key, element: element})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(zRemArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) ZRemRangeByScore(ctx context.Context, key, min, max string) (int64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, zRemRangeByScoreCmd, zRemRangeByScoreArgs{key: key, min: min, max: max})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(zRemRangeByScoreArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) ZScore(ctx context.Context, key, element string) (float64, error) {
	key = prefixKey(key, r.Opts.KeyPrefix)
	arg, err := r.cmd(ctx, zScoreCmd, zScoreArgs{key: key, element: element})
	if err != nil {
		return 0, err
	}
	args, ok := arg.(zScoreArgs)
	if !ok {
		return 0, rc.ErrInvalidArguments
	}
	return args.res, nil
}

func (r *rediser) cmd(ctx context.Context, cmd string, args interface{}) (interface{}, error) {
	sub, startTime := rc.XactStart(ctx)
	res, err := r.cmdProcessor(ctx, cmd, args)
	dur := rc.XactEnd(sub, startTime)
	r.report(cmd, err == nil || err == redis.Nil, dur)
	return res, err
}

func (r *rediser) cmdProcessor(ctx context.Context, cmd string, args interface{}) (interface{}, error) {
	active := atomic.AddInt32(&r.active, 1)
	defer atomic.AddInt32(&r.active, -1)
	r.lock.Lock()
	if active > r.maxActive {
		r.maxActive = active
	}
	r.lock.Unlock()

	// If the context has been canceled, don't send the command to Redis.
	ctx, cancel := context.WithCancel(ctx)
	defer cancel()
	select {
	case <-ctx.Done():
		return nil, ctx.Err()
	default:
		break
	}

	// FIXME: Consider getting rid of the case statement while maintaining the
	// reusability of the stats code (maxActive and r.report()).
	var out interface{}
	switch cmd {
	case delCmd:
		arg, ok := args.(delArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.Del(arg.keys...).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case evalCmd:
		arg, ok := args.(evalArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.Eval(arg.script, arg.keys, arg.args...).Result()
		if err != nil {
			return arg, err
		}
		out = val
	case existsCmd:
		arg, ok := args.(existsArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.Exists(arg.keys...).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case expireCmd:
		arg, ok := args.(expireArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.Expire(arg.key, arg.ttl).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case flushDbCmd:
		if err := r.Client.FlushDb().Err(); err != nil {
			return nil, err
		}
		out = nil
	case getCmd:
		arg, ok := args.(getArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.Get(arg.key).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case hDelCmd:
		arg, ok := args.(hDelArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.HDel(arg.key, arg.fields...).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case hGetAllCmd:
		arg, ok := args.(hGetAllArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.HGetAll(arg.key).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case hKeysCmd:
		arg, ok := args.(hKeysArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.HKeys(arg.key).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case hMGetCmd:
		arg, ok := args.(hMGetArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.HMGet(arg.key, arg.fields...).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case hMSetCmd:
		arg, ok := args.(hMSetArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		_, err := r.Client.HMSet(arg.key, arg.fields).Result()
		if err != nil {
			return arg, err
		}
		out = arg
	case hSetCmd:
		arg, ok := args.(hSetArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.HSet(arg.key, arg.field, arg.val).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case hSetNXCmd:
		arg, ok := args.(hSetNXArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.HSetNX(arg.key, arg.field, arg.val).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case hIncrByCmd:
		arg, ok := args.(hIncrByArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.HIncrBy(arg.key, arg.field, arg.val).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case hLenCmd:
		arg, ok := args.(hLenArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.HLen(arg.key).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case incrCmd:
		arg, ok := args.(incrArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.Incr(arg.key).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case lLenCmd:
		arg, ok := args.(lLenArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.LLen(arg.key).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case lRangeCmd:
		arg, ok := args.(lRangeArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.LRange(arg.key, arg.start, arg.stop).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case lRemCmd:
		arg, ok := args.(lRemArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.LRem(arg.key, arg.count, arg.val).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case lTrimCmd:
		arg, ok := args.(lTrimArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		_, err := r.Client.LTrim(arg.key, arg.start, arg.stop).Result()
		if err != nil {
			return arg, err
		}
		out = arg
	case mGetCmd:
		arg, ok := args.(mGetArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.MGet(arg.keys...).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case mSetNxTTLCmd:
		arg, ok := args.(mSetTTLArgs)
		if !ok || len(arg.pairs)%2 != 0 {
			return arg, rc.ErrInvalidArguments
		}
		p := r.Client.Pipeline()
		defer func() {
			_ = p.Close()
		}()
		for i := 0; i < len(arg.pairs); i += 2 {
			p.SetNX(arg.pairs[i].(string), arg.pairs[i+1], arg.ttl)
		}
		_, err := p.Exec()

		if err != nil {
			return arg, err
		}
		out = arg
	case mSetCmd:
		arg, ok := args.(mSetArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		_, err := r.Client.MSet(arg.pairs...).Result()
		if err != nil {
			return arg, err
		}
		out = arg
	case mSetTTLCmd:
		arg, ok := args.(mSetTTLArgs)
		if !ok || len(arg.pairs)%2 != 0 {
			return arg, rc.ErrInvalidArguments
		}
		p := r.Client.Pipeline()
		defer func() {
			_ = p.Close()
		}()
		for i := 0; i < len(arg.pairs); i += 2 {
			p.Set(arg.pairs[i].(string), arg.pairs[i+1], arg.ttl)
		}
		_, err := p.Exec()

		if err != nil {
			return arg, err
		}
		out = arg
	case pipelinedDelCmd:
		arg, ok := args.(pipelinedDelArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		p := r.Client.Pipeline()
		defer func() {
			_ = p.Close()
		}()
		intCmds := make([]*redis.IntCmd, len(arg.keys))
		for i, key := range arg.keys {
			intCmds[i] = p.Del(key)
		}
		_, err := p.Exec()
		if err != nil {
			return arg, err
		}
		var x int64
		for _, intCmd := range intCmds {
			x += intCmd.Val()
		}
		arg.res = x
		out = arg

	case pipelinedGetCmd:
		arg, ok := args.(pipelinedGetArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		p := r.Client.Pipeline()
		defer func() {
			_ = p.Close()
		}()
		stringCmds := make([]*redis.StringCmd, len(arg.keys))
		for i, key := range arg.keys {
			stringCmds[i] = p.Get(key)
		}
		_, err := p.Exec()
		if err != nil && err != redis.Nil {
			return arg, err
		}
		response := make([]interface{}, len(arg.keys))
		for i, stringCmd := range stringCmds {
			response[i] = stringCmd.Val()
		}
		arg.res = response
		out = arg
	case rPushCmd:
		arg, ok := args.(rPushArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.RPush(arg.key, arg.val).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case sAddCmd:
		arg, ok := args.(sAddArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.SAdd(arg.key, arg.vals...).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case sDiffCmd:
		arg, ok := args.(sDiffArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.SDiff(arg.keys...).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case setCmd:
		arg, ok := args.(setArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		if _, err := r.Client.Set(arg.key, arg.val, arg.ttl).Result(); err != nil {
			return arg, err
		}
		out = arg
	case setNxCmd:
		arg, ok := args.(setNxArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.SetNX(arg.key, arg.val, arg.ttl).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case sIsMemberCmd:
		arg, ok := args.(sIsMemberArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.SIsMember(arg.key, arg.val).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case sMembersCmd:
		arg, ok := args.(sMembersArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.SMembers(arg.key).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case sRemCmd:
		arg, ok := args.(sRemArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.SRem(arg.key, arg.vals...).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case sScanCmd:
		arg, ok := args.(sScanArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		vals, cursor, err := r.Client.SScan(arg.key, arg.cursor, arg.match, arg.count).Result()
		if err != nil {
			return arg, err
		}
		arg.resCursor = cursor
		arg.res = vals
		out = arg
	case ttlCmd:
		arg, ok := args.(ttlArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.TTL(arg.key).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case zAddCmd:
		arg, ok := args.(zAddArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.ZAdd(arg.key, arg.items...).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case zCardCmd:
		arg, ok := args.(zCardArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.ZCard(arg.key).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case zRangeByLexCmd:
		arg, ok := args.(zRangeByLexArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.ZRangeByLex(arg.key, arg.opt).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case zRangeByScoreCmd:
		arg, ok := args.(zRangeByScoreArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.ZRangeByScore(arg.key, arg.opt).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case zRemCmd:
		arg, ok := args.(zRemArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.ZRem(arg.key, arg.element).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case zRemRangeByScoreCmd:
		arg, ok := args.(zRemRangeByScoreArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.ZRemRangeByScore(arg.key, arg.min, arg.max).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	case zScoreCmd:
		arg, ok := args.(zScoreArgs)
		if !ok {
			return arg, rc.ErrInvalidArguments
		}
		val, err := r.Client.ZScore(arg.key, arg.element).Result()
		if err != nil {
			return arg, err
		}
		arg.res = val
		out = arg
	default:
		return nil, rc.ErrInvalidCommand
	}
	return out, nil
}

func (r *rediser) report(stat string, success bool, dur time.Duration) {
	var result string
	if success {
		result = successStr
	} else {
		result = errorStr
	}
	bucket := fmt.Sprintf("%s.%s.%s", r.Opts.StatPrefix, stat, result)
	err := r.Stats.Inc(bucket, 1, r.Opts.StatSampleRate)
	if err != nil {
		log.Printf("failed to report statsd metric %s: %q", bucket, err)
	}
	// FIXME: Allow configuration of sample rate.
	err = r.Stats.TimingDuration(bucket, dur, 0.1)
	if err != nil {
		log.Printf("failed to report statsd metric %s: %q", bucket, err)
	}
}

func prefixKey(key string, prefix string) string {
	if prefix == "" {
		return key
	}
	return fmt.Sprintf("%s:%s", prefix, key)
}

func prefixKeys(keys []string, prefix string) []string {
	if prefix == "" {
		return keys
	}
	out := make([]string, len(keys))
	for i := range keys {
		out[i] = prefixKey(keys[i], prefix)
	}
	return out
}

// prefixPairs processes a slice of interface{}, where every even-index element
// is a key and every odd-index element is a value. Skip every other element to
// prefix only the keys.
func prefixPairs(pairs []interface{}, prefix string) ([]interface{}, error) {
	if prefix == "" {
		return pairs, nil
	}
	for i := 0; i < len(pairs); i += 2 {
		s, ok := pairs[i].(string)
		if !ok {
			return pairs, fmt.Errorf("prefixPairs: cannot prefix non-string key %v", pairs[i])
		}
		pairs[i] = prefixKey(s, prefix)
	}
	return pairs, nil
}
