package rediser

import (
	"context"
	"reflect"
	"time"
)

// MockHandlerWithStubbedCache wraps the MockHandler mock but stubs out all the
// cache helper functions so they always cache miss. This is useful for tests
// that don't want to test caching behavior.
type MockHandlerWithStubbedCache struct {
	MockHandler
}

// Cached behaves as a cache miss.
func (m *MockHandlerWithStubbedCache) Cached(ctx context.Context, key string, ttl time.Duration, res interface{}, fn func() (interface{}, error)) error {
	fetched, err := fn()
	if err != nil {
		return err
	}

	rv := reflect.ValueOf(res).Elem()
	if fetched == nil {
		rv.Set(reflect.Zero(rv.Type()))
	} else {
		rv.Set(reflect.ValueOf(fetched))
	}
	return nil
}

// CachedBool behaves as a cache miss.
func (m *MockHandlerWithStubbedCache) CachedBool(ctx context.Context, key string, ttl time.Duration, fn func() (interface{}, error)) (bool, error) {
	val, err := fn()
	if err != nil {
		return false, err
	}
	return val.(bool), nil
}

// CachedBoolDifferentTTLs behaves as a cache miss.
func (m *MockHandlerWithStubbedCache) CachedBoolDifferentTTLs(ctx context.Context, key string, ttls CachedBoolTTLs, fn func() (bool, error)) (bool, error) {
	return fn()
}

// CachedInt behaves as a cache miss.
func (m *MockHandlerWithStubbedCache) CachedInt(ctx context.Context, key string, ttl time.Duration, fn func() (interface{}, error)) (int, error) {
	val, err := fn()
	if err != nil {
		return 0, err
	}
	return val.(int), nil
}

// CachedInt64 behaves as a cache miss.
func (m *MockHandlerWithStubbedCache) CachedInt64(ctx context.Context, key string, ttl time.Duration, fn func() (interface{}, error)) (int64, error) {
	val, err := fn()
	if err != nil {
		return int64(0), err
	}
	return val.(int64), nil
}

// CachedString behaves as a cache miss.
func (m *MockHandlerWithStubbedCache) CachedString(ctx context.Context, key string, ttl time.Duration, fn func() (interface{}, error)) (string, error) {
	val, err := fn()
	if err != nil {
		return "", err
	}
	return val.(string), nil
}

// CachedIntSlice behaves as a cache miss.
func (m *MockHandlerWithStubbedCache) CachedIntSlice(ctx context.Context, key string, ttl time.Duration, fn func() (interface{}, error)) ([]int, error) {
	val, err := fn()
	if err != nil {
		return nil, err
	}
	return val.([]int), nil
}

// CachedInt64Slice behaves as a cache miss.
func (m *MockHandlerWithStubbedCache) CachedInt64Slice(ctx context.Context, key string, ttl time.Duration, fn func() (interface{}, error)) ([]int64, error) {
	val, err := fn()
	if err != nil {
		return nil, err
	}
	return val.([]int64), nil
}

// CachedStringSlice behaves as a cache miss.
func (m *MockHandlerWithStubbedCache) CachedStringSlice(ctx context.Context, key string, ttl time.Duration, fn func() (interface{}, error)) ([]string, error) {
	val, err := fn()
	if err != nil {
		return nil, err
	}
	return val.([]string), nil
}
