package common

import (
	"encoding/json"
	"fmt"
	"time"

	"github.com/go-redis/redis"
)

// Options specifies the configuration for initializing a Redis cache client
// Cluster options: https://godoc.org/github.com/go-redis/redis#ClusterOptions
// Ring options: https://godoc.org/github.com/go-redis/redis#RingOptions
type Options struct {
	// Redis options
	Addrs              []string // required
	DialTimeout        time.Duration
	ReadTimeout        time.Duration
	WriteTimeout       time.Duration
	PoolSize           int
	PoolTimeout        time.Duration
	IdleTimeout        time.Duration
	IdleCheckFrequency time.Duration
	Password           string
	KeyPrefix          string

	// Cluster options
	MaxRedirects    int
	UseReadReplicas bool
	RouteByLatency  bool

	// Ring options
	HeartbeatFrequency time.Duration
	DB                 int
	MaxRetries         int

	// Rediser options
	StatPrefix      string        // required
	StatSampleRate  float32       // required
	MonitorInterval time.Duration // required

	// Cache-specific options
	MarshalFunc   func(interface{}) ([]byte, error)
	UnmarshalFunc func([]byte, interface{}) error
}

var (
	marshalFunc = func(v interface{}) ([]byte, error) {
		return json.Marshal(v)
	}
	unmarshalFunc = func(b []byte, v interface{}) error {
		return json.Unmarshal(b, v)
	}
)

// Validate checks to make sure required fields are set.
func (opts *Options) Validate() error {
	if len(opts.Addrs) == 0 {
		return fmt.Errorf("%q: redis addresses", ErrInvalidRedisSetting)
	}
	if opts.StatPrefix == "" {
		return fmt.Errorf("%q: stat prefix", ErrInvalidRedisSetting)
	}
	if opts.StatSampleRate <= 0 {
		return fmt.Errorf("%q: stat sample rate", ErrInvalidRedisSetting)
	}
	if opts.MonitorInterval <= 0 {
		return fmt.Errorf("%q: monitor interval", ErrInvalidRedisSetting)
	}

	if opts.MarshalFunc == nil {
		opts.MarshalFunc = marshalFunc
	}
	if opts.UnmarshalFunc == nil {
		opts.UnmarshalFunc = unmarshalFunc
	}

	return nil
}

// GetClusterOptions converts our generic Options into Cluster-specific options.
func (opts *Options) GetClusterOptions() *redis.ClusterOptions {
	return &redis.ClusterOptions{
		Addrs:              opts.Addrs,
		MaxRedirects:       opts.MaxRedirects,
		ReadOnly:           opts.UseReadReplicas,
		RouteByLatency:     opts.RouteByLatency,
		Password:           opts.Password,
		DialTimeout:        opts.DialTimeout,
		ReadTimeout:        opts.ReadTimeout,
		WriteTimeout:       opts.WriteTimeout,
		PoolSize:           opts.PoolSize,
		PoolTimeout:        opts.PoolTimeout,
		IdleTimeout:        opts.IdleTimeout,
		IdleCheckFrequency: opts.IdleCheckFrequency,
	}
}

// GetRingOptions converts our generic Options into Ring-specific options.
func (opts *Options) GetRingOptions() *redis.RingOptions {
	addrs := make(map[string]string, len(opts.Addrs))
	for _, addr := range opts.Addrs {
		addrs[addr] = addr
	}
	return &redis.RingOptions{
		Addrs:              addrs,
		HeartbeatFrequency: opts.HeartbeatFrequency,
		DB:                 opts.DB,
		Password:           opts.Password,
		MaxRetries:         opts.MaxRetries,
		DialTimeout:        opts.DialTimeout,
		ReadTimeout:        opts.ReadTimeout,
		WriteTimeout:       opts.WriteTimeout,
		PoolSize:           opts.PoolSize,
		PoolTimeout:        opts.PoolTimeout,
		IdleTimeout:        opts.IdleTimeout,
		IdleCheckFrequency: opts.IdleCheckFrequency,
	}
}
