package blacklist

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/common/twitchhttp"
	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"
)

type AddIPBody struct {
	ID       string `json:"id"`
	Duration string `json:"duration_in_seconds,omitempty"`
}

type validateIPResp struct {
	BlackListed bool `json:"blacklisted"`
}

type ErrorResponse struct {
	Status  int
	Message string
	Error   string
}

// BlacklistInterface exposes methods to interact with the blacklist
type Client interface {
	AddIP(ctx context.Context, IP string, durationInSeconds string, reqOpts *twitchhttp.ReqOpts) error
	DeleteIP(ctx context.Context, IP string, reqOpts *twitchhttp.ReqOpts) error
	ValidateIP(ctx context.Context, IP string, reqOpts *twitchhttp.ReqOpts) (bool, error)
}

// NewBlacklistClient creates a Blacklist client
func NewClient(host string, stats statsd.Statter) (Client, error) {
	c, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host:           host,
		Stats:          stats,
		TimingXactName: "service.blacklist",
	})
	if err != nil {
		return nil, err
	}

	return &blacklistImpl{
		Client: c,
	}, nil
}

type blacklistImpl struct {
	twitchhttp.Client
}

// AddIP adds an IP address to the global blacklist
func (b *blacklistImpl) AddIP(ctx context.Context, IP string, durationInSeconds string, reqOpts *twitchhttp.ReqOpts) error {
	endpoint := "/api/internal/blacklisted_ips/"

	var params AddIPBody
	params.ID = IP

	if durationInSeconds != "" {
		params.Duration = durationInSeconds
	}

	bodyJson, err := json.Marshal(params)
	if err != nil {
		return err
	}

	body := bytes.NewBuffer(bodyJson)

	req, err := b.NewRequest("POST", endpoint, body)
	if err != nil {
		return err
	}
	req.Header.Set("Content-Type", "application/json")

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "add_ip",
		StatSampleRate: 1.0,
	})

	resp, err := b.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode >= 400 {
		return b.handleFailedRequest(resp)
	}
	return nil
}

// DeleteIP removes an IP address from the global blacklist
func (b *blacklistImpl) DeleteIP(ctx context.Context, IP string, reqOpts *twitchhttp.ReqOpts) error {
	epFormat := "/api/internal/blacklisted_ips/%s"
	endpoint := fmt.Sprintf(epFormat, IP)

	req, err := b.NewRequest("DELETE", endpoint, nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "delete_ip",
		StatSampleRate: 1.0,
	})

	resp, err := b.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer resp.Body.Close()

	if resp.StatusCode >= 400 {
		return b.handleFailedRequest(resp)
	}
	return nil
}

// Validate if an IP is in the global blacklist
func (b *blacklistImpl) ValidateIP(ctx context.Context, IP string, reqOpts *twitchhttp.ReqOpts) (bool, error) {
	epFormat := "/api/internal/blacklisted_ips/validate?id=%s"
	endpoint := fmt.Sprintf(epFormat, IP)

	req, err := b.NewRequest("GET", endpoint, nil)
	if err != nil {
		return false, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "validate_ip",
		StatSampleRate: 1.0,
	})

	resp, err := b.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return false, err
	}
	defer resp.Body.Close()

	if resp.StatusCode >= 400 {
		return false, b.handleFailedRequest(resp)
	}

	var decoded validateIPResp
	if err := json.NewDecoder(resp.Body).Decode(&decoded); err != nil {
		return false, err
	}
	return decoded.BlackListed, nil
}

// handleFailedRequest is a generic error parser / formatter for all blacklist endpoints
func (b *blacklistImpl) handleFailedRequest(resp *http.Response) error {
	var errResp ErrorResponse
	err := json.NewDecoder(resp.Body).Decode(&errResp)
	if err != nil {
		return fmt.Errorf("Failed blacklist request. StatusCode=%v Unable to read response body (%v)", resp.StatusCode, err)
	}

	return fmt.Errorf(`Failed blacklist request. StatusCode=%v Message=%v`, resp.StatusCode, errResp.Message)
}
