package discovery

import (
	"net/url"
	"strings"

	"strconv"

	"code.justin.tv/common/twitchhttp"
	"golang.org/x/net/context"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "discovery"
)

type Client interface {
	Get(ctx context.Context, id, locale string, reqOpts *twitchhttp.ReqOpts) (*LocalizedGame, error)
	GetAll(ctx context.Context, ids []string, locale string, limit, offset int, reqOpts *twitchhttp.ReqOpts) (map[string]*LocalizedGame, error)
	GetByName(ctx context.Context, name, locale string, reqOpts *twitchhttp.ReqOpts) (*LocalizedGame, error)
	GetAllAliasesByID(ctx context.Context, id string, reqOpts *twitchhttp.ReqOpts) ([]string, error)
	GetAllAliasesByName(ctx context.Context, name string, reqOpts *twitchhttp.ReqOpts) ([]string, error)
	Suggest(ctx context.Context, term, locale string, params *SuggestParams, reqOpts *twitchhttp.ReqOpts) ([]LocalizedGame, error)
	GetLocalizationsByID(ctx context.Context, gameID, locale string, reqOpts *twitchhttp.ReqOpts) (*Localizations, error)
	GetLocalizationsByName(ctx context.Context, gameName, locale string, reqOpts *twitchhttp.ReqOpts) (*Localizations, error)
	AddLocalizationByID(ctx context.Context, gameID, locale, localizedName string, reqOpts *twitchhttp.ReqOpts) (*Localizations, error)
	AddLocalizationByName(ctx context.Context, gameName, locale, localizedName string, reqOpts *twitchhttp.ReqOpts) (*Localizations, error)
	DeleteLocalizationByID(ctx context.Context, gameID, locale string, reqOpts *twitchhttp.ReqOpts) error
	DeleteLocalizationByName(ctx context.Context, gameName, locale string, reqOpts *twitchhttp.ReqOpts) error
	DeleteAllLocalizationsByID(ctx context.Context, gameID string, reqOpts *twitchhttp.ReqOpts) error
	DeleteAllLocalizationsByName(ctx context.Context, gameName string, reqOpts *twitchhttp.ReqOpts) error
}

type client struct {
	twitchhttp.Client
}

func NewClient(conf twitchhttp.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchhttp.NewClient(conf)
	return &client{twitchClient}, err
}

func (c *client) Get(ctx context.Context, id, locale string, reqOpts *twitchhttp.ReqOpts) (*LocalizedGame, error) {
	return c.get(ctx, "id", id, locale, "service.discovery.get", reqOpts)
}

func (c *client) GetByName(ctx context.Context, name, locale string, reqOpts *twitchhttp.ReqOpts) (*LocalizedGame, error) {
	return c.get(ctx, "name", name, locale, "service.discovery.get_by_name", reqOpts)
}

func (c *client) Suggest(ctx context.Context, term, locale string, params *SuggestParams, reqOpts *twitchhttp.ReqOpts) ([]LocalizedGame, error) {
	query := url.Values{
		"term":   {term},
		"locale": {locale},
	}
	if params != nil {
		if params.Live {
			query.Add("live", "true")
		}
	}

	req, err := c.NewRequest("GET", "/suggest?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.discovery.suggest",
		StatSampleRate: defaultStatSampleRate,
	})

	var data []LocalizedGame
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

func (c *client) GetAll(ctx context.Context, ids []string, locale string, limit, offset int, reqOpts *twitchhttp.ReqOpts) (map[string]*LocalizedGame, error) {
	query := url.Values{
		"id":     {strings.Join(ids, ",")},
		"limit":  {strconv.Itoa(limit)},
		"offset": {strconv.Itoa(offset)},
		"locale": {locale},
	}

	req, err := c.NewRequest("GET", "/game/list?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.discovery.get_all",
		StatSampleRate: defaultStatSampleRate,
	})

	var data []*LocalizedGame
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	resp := make(map[string]*LocalizedGame)
	for _, game := range data {
		resp[strconv.Itoa(game.Game.ID)] = game
	}
	return resp, nil
}

func (c *client) GetAllAliasesByID(ctx context.Context, id string, reqOpts *twitchhttp.ReqOpts) ([]string, error) {
	return c.getAllAliases(ctx, "id", id, "service.discovery.get_all_aliases_by_id", reqOpts)
}

func (c *client) GetAllAliasesByName(ctx context.Context, name string, reqOpts *twitchhttp.ReqOpts) ([]string, error) {
	return c.getAllAliases(ctx, "name", name, "service.discovery.get_all_aliases_by_name", reqOpts)
}

func (c *client) getAllAliases(ctx context.Context, key, value, statName string, reqOpts *twitchhttp.ReqOpts) ([]string, error) {
	query := url.Values{
		key: {value},
	}

	req, err := c.NewRequest("GET", "/aliases?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       statName,
		StatSampleRate: defaultStatSampleRate,
	})

	var data Aliases
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return data.Aliases, err
}

func (c *client) GetLocalizationsByID(ctx context.Context, gameID, locale string, reqOpts *twitchhttp.ReqOpts) (*Localizations, error) {
	return c.getLocalizations(ctx, "game_id", gameID, locale, "service.discovery.get_localizations_by_id", reqOpts)
}

func (c *client) GetLocalizationsByName(ctx context.Context, gameName, locale string, reqOpts *twitchhttp.ReqOpts) (*Localizations, error) {
	return c.getLocalizations(ctx, "game_name", gameName, locale, "service.discovery.get_localizations_by_name", reqOpts)
}

func (c *client) AddLocalizationByID(ctx context.Context, gameID, locale, localizedName string, reqOpts *twitchhttp.ReqOpts) (*Localizations, error) {
	return c.addLocalization(ctx, "game_id", gameID, locale, localizedName, "service.discovery.add_localization_by_id", reqOpts)
}

func (c *client) AddLocalizationByName(ctx context.Context, gameName, locale, localizedName string, reqOpts *twitchhttp.ReqOpts) (*Localizations, error) {
	return c.addLocalization(ctx, "game_name", gameName, locale, localizedName, "service.discovery.add_localization_by_name", reqOpts)
}

func (c *client) DeleteLocalizationByID(ctx context.Context, gameID, locale string, reqOpts *twitchhttp.ReqOpts) error {
	return c.deleteLocalization(ctx, "game_id", gameID, locale, "service.discovery.delete_localizations_by_id", reqOpts)
}

func (c *client) DeleteLocalizationByName(ctx context.Context, gameName, locale string, reqOpts *twitchhttp.ReqOpts) error {
	return c.deleteLocalization(ctx, "game_name", gameName, locale, "service.discovery.delete_localizations_by_name", reqOpts)
}

func (c *client) DeleteAllLocalizationsByID(ctx context.Context, gameID string, reqOpts *twitchhttp.ReqOpts) error {
	return c.deleteAllLocalizations(ctx, "game_id", gameID, "service.discovery.delete_all_localizations_by_id", reqOpts)
}

func (c *client) DeleteAllLocalizationsByName(ctx context.Context, gameName string, reqOpts *twitchhttp.ReqOpts) error {
	return c.deleteAllLocalizations(ctx, "game_name", gameName, "service.discovery.delete_all_localizations_by_name", reqOpts)
}

func (c *client) get(ctx context.Context, key, value, locale, stat string, reqOpts *twitchhttp.ReqOpts) (*LocalizedGame, error) {
	query := url.Values{
		key:      {value},
		"locale": {locale},
	}

	req, err := c.NewRequest("GET", "/game?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       stat,
		StatSampleRate: defaultStatSampleRate,
	})

	var data *LocalizedGame
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

func (c *client) getLocalizations(ctx context.Context, key, value, locale, stat string, reqOpts *twitchhttp.ReqOpts) (*Localizations, error) {
	query := url.Values{
		key:      {value},
		"locale": {locale},
	}

	req, err := c.NewRequest("GET", "/localizations?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       stat,
		StatSampleRate: defaultStatSampleRate,
	})

	var data *Localizations
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

func (c *client) addLocalization(ctx context.Context, key, value, locale, localizedName, stat string, reqOpts *twitchhttp.ReqOpts) (*Localizations, error) {
	form := url.Values{
		key:              {value},
		"locale":         {locale},
		"localized_name": {localizedName},
	}

	req, err := c.NewRequest("PUT", "/localizations", strings.NewReader(form.Encode()))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       stat,
		StatSampleRate: defaultStatSampleRate,
	})

	var data *Localizations
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

func (c *client) deleteLocalization(ctx context.Context, key, value, locale, stat string, reqOpts *twitchhttp.ReqOpts) error {
	query := url.Values{
		key:      {value},
		"locale": {locale},
	}

	req, err := c.NewRequest("DELETE", "/localizations?"+query.Encode(), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       stat,
		StatSampleRate: defaultStatSampleRate,
	})

	_, err = c.DoJSON(ctx, nil, req, combinedReqOpts)
	return err
}

func (c *client) deleteAllLocalizations(ctx context.Context, key, value, stat string, reqOpts *twitchhttp.ReqOpts) error {
	query := url.Values{
		key: {value},
	}

	req, err := c.NewRequest("DELETE", "/localizations/all?"+query.Encode(), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       stat,
		StatSampleRate: defaultStatSampleRate,
	})

	_, err = c.DoJSON(ctx, nil, req, combinedReqOpts)
	return err
}
