package owl

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/url"

	"golang.org/x/net/context"

	"strconv"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/web/owl/oauth2"
)

// errors
var (
	ErrInvalidClientID     = errors.New("Invalid client id")
	ErrInvalidClientRowID  = errors.New("Invalid client row id")
	ErrInvalidClientSecret = errors.New("Invalid client secret")
	ErrNoAuthorization     = errors.New("No authorization found for the given user and client")
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "owl"
)

type SortKey string
type SortOrder string
type FilterableColumn string

const (
	// SortKeyClientName is the sort key used to sort on the client's name
	SortKeyClientName SortKey = "name"

	// SortKeyClientID is the sort key used to sort on a client's client_id (i.e. the API key)
	SortKeyClientID SortKey = "client_id"

	// SortAsc specifies that the sort is ordered from lowest to highest
	SortAsc SortOrder = "ASC"

	// SortDesc specifies that the sort is ordered from highest to lowest
	SortDesc SortOrder = "DESC"

	// FilterColName is the key used to filter queries by the "name" column
	FilterColName FilterableColumn = "name"

	// FilterColOwnerID is the key used to filter queries by the "oauth2_client_owner_id" column
	FilterColOwnerID FilterableColumn = "owner_id"
)

type Client interface {
	Authorize(ctx context.Context, authReq url.Values, reqOpts *twitchhttp.ReqOpts) (Token, error)
	Validate(ctx context.Context, token string, scopes []string, reqOpts *twitchhttp.ReqOpts) (*oauth2.Authorization, error)
	ClientInfo(ctx context.Context, clientID string, reqOpts *twitchhttp.ReqOpts) (*ClientData, error)
	Authorizations(ctx context.Context, ownerID string, reqOpts *twitchhttp.ReqOpts) (*AuthorizationsResponse, error)
	GetClient(ctx context.Context, clientID string, reqOpts *twitchhttp.ReqOpts) (*oauth2.Client, error)
	GetClients(ctx context.Context, cursor string, showHidden bool,
		filters map[FilterableColumn]string, sKey SortKey, sOrder SortOrder,
		reqOpts *twitchhttp.ReqOpts) ([]*oauth2.Client, string, error)
	GetClientByRowID(ctx context.Context, clientRowID string, reqOpts *twitchhttp.ReqOpts) (*oauth2.Client, error)
	ValidateSecret(ctx context.Context, clientID, clientSecret string, reqOpts *twitchhttp.ReqOpts) (bool, error)
	ExchangeAuthorizationCode(ctx context.Context, clientID, clientSecret, code, redirectURI string, reqOpts *twitchhttp.ReqOpts) (*TokenResponse, error)
	CreateAuthorization(ctx context.Context, ownerID, clientID string, scopes []string, reqOpts *twitchhttp.ReqOpts) (*TokenResponse, error)
	DeleteAllSessions(ctx context.Context, ownerID string, reqOpts *twitchhttp.ReqOpts) error
}

type client struct {
	twitchhttp.Client
}

type ErrorResponse struct {
	Status  int    `json:"status"`
	Message string `json:"message"`
	Error   string `json:"error"`
}

type ClientData struct {
	RowID string `json:"client_row_id"`
}

type getClientsResponse struct {
	Clients    []*oauth2.Client `json:"clients"`
	NextCursor string           `json:"next_cursor"`
}

func NewClient(conf twitchhttp.ClientConf) (Client, error) {
	confWithXactName := addXactNameIfNoneExists(conf)
	twitchClient, err := twitchhttp.NewClient(confWithXactName)
	if err != nil {
		return nil, err
	}

	return &client{twitchClient}, nil
}

func addXactNameIfNoneExists(conf twitchhttp.ClientConf) twitchhttp.ClientConf {
	if conf.TimingXactName != "" {
		return conf
	}
	conf.TimingXactName = defaultTimingXactName
	return conf
}

// Validate makes an API call to Owl to lookup the authorization for a token.
// If the token is invalid or doesn't have a superset of scopes, returns an authorziation with Valid=false
func (c *client) Validate(ctx context.Context, token string, scopes []string, reqOpts *twitchhttp.ReqOpts) (*oauth2.Authorization, error) {
	scope := oauth2.BuildScope(scopes)
	query := url.Values{
		"token": {token},
	}
	if len(scope) > 0 {
		query.Add("scope", scope)
	}

	req, err := c.NewRequest("GET", "/validate?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.owl.validate",
		StatSampleRate: defaultStatSampleRate,
	})

	var data *oauth2.Authorization
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

// AuthResponse respresents a response from Owl's authorize endpoint
type AuthResponse struct {
	ErrorResponse
	Redirect string `json:"redirect"`
}

// Authorize extends owl's http interface to create/read authorizations
// between a requesting OAuth Client and an end user
func (c *client) Authorize(ctx context.Context, authReq url.Values, reqOpts *twitchhttp.ReqOpts) (Token, error) {

	req, err := c.NewRequest("POST", "/authorize", bytes.NewBufferString(authReq.Encode()))
	if err != nil {
		return nil, err
	}

	req.Header.Add("Content-Type", "application/x-www-form-urlencoded")

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.owl.authorize",
		StatSampleRate: defaultStatSampleRate,
	})

	var res *AuthResponse
	_, err = c.DoJSON(ctx, &res, req, combinedReqOpts)
	if err != nil {
		if twitchhttpErr, ok := err.(*twitchhttp.Error); ok {
			if twitchhttpErr.StatusCode == http.StatusUnauthorized {
				return nil, ErrNoAuthorization
			}
		}
		return nil, err
	}

	resURI, err := url.Parse(res.Redirect)
	if err != nil {
		return nil, err
	}

	if resURI.RawQuery != "" {
		query := resURI.Query()
		if errCode := query.Get("error"); errCode != "" {
			return nil, &Error{Code: errCode, Message: query.Get("error_description")}
		}
		return &tokenImpl{code: query.Get("code"), scope: query.Get("scope")}, nil
	}

	fragment, err := url.ParseQuery(resURI.Fragment)
	if err != nil {
		return nil, err
	}

	return &tokenImpl{accessToken: fragment.Get("access_token"), scope: fragment.Get("scope")}, nil
}

func (c *client) GetClient(ctx context.Context, clientID string, reqOpts *twitchhttp.ReqOpts) (*oauth2.Client, error) {
	if clientID == "" {
		return nil, &twitchhttp.Error{
			StatusCode: 400,
			Message:    "No client ID specified",
		}
	}

	req, err := c.NewRequest("GET", "/clients/"+clientID, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.owl.get_client",
		StatSampleRate: defaultStatSampleRate,
	})

	client := &oauth2.Client{}
	_, err = c.DoJSON(ctx, &client, req, combinedReqOpts)
	if twitchhttpErr, ok := err.(*twitchhttp.Error); ok {
		if twitchhttpErr.StatusCode == http.StatusNotFound {
			return nil, ErrInvalidClientID
		}
	}

	return client, err
}

func (c *client) GetClientByRowID(ctx context.Context, clientRowID string, reqOpts *twitchhttp.ReqOpts) (*oauth2.Client, error) {
	req, err := c.NewRequest("GET", "/clients_legacy/"+clientRowID, nil)
	if err != nil {
		return nil, err
	}
	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.owl.get_client_by_row_id",
		StatSampleRate: defaultStatSampleRate,
	})

	client := &oauth2.Client{}
	_, err = c.DoJSON(ctx, &client, req, combinedReqOpts)
	if twitchhttpErr, ok := err.(*twitchhttp.Error); ok {
		if twitchhttpErr.StatusCode == http.StatusNotFound {
			return nil, ErrInvalidClientRowID
		}
	}

	return client, err
}

// GetClients returns a slice of clients that passes the filters provided and a cursor
// to continue enumerating the rest of the matching result set if it is larger than the
// slice returned.
func (c *client) GetClients(ctx context.Context, cursor string, showHidden bool,
	filters map[FilterableColumn]string, sKey SortKey, sOrder SortOrder,
	reqOpts *twitchhttp.ReqOpts) ([]*oauth2.Client, string, error) {
	params := url.Values{}
	params.Set("show_hidden", strconv.FormatBool(showHidden))
	params.Set("cursor", cursor)
	for col, val := range filters {
		params.Set(string(col), val)
	}

	query := params.Encode()
	req, err := c.NewRequest("GET", "/clients?"+query, nil)
	if err != nil {
		return nil, "", err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.owl.get_all_clients",
		StatSampleRate: defaultStatSampleRate,
	})
	res := &getClientsResponse{}
	_, err = c.DoJSON(ctx, &res, req, combinedReqOpts)

	if err != nil {
		return nil, "", err
	}
	return res.Clients, res.NextCursor, nil
}

func (c *client) ClientInfo(ctx context.Context, clientID string, reqOpts *twitchhttp.ReqOpts) (*ClientData, error) {
	if clientID == "" {
		return nil, &twitchhttp.Error{
			StatusCode: 400,
			Message:    "No client ID specified",
		}
	}

	query := url.Values{"client_id": {clientID}}
	req, err := c.NewRequest("GET", "/client_info?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.owl.client_info",
		StatSampleRate: defaultStatSampleRate,
	})

	var data *ClientData
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if twitchhttpErr, ok := err.(*twitchhttp.Error); ok {
		if twitchhttpErr.StatusCode == http.StatusNotFound {
			return nil, ErrInvalidClientID
		}
	}

	return data, err
}

type AuthorizationsResponse struct {
	Authorizations []*oauth2.Authorization `json:"authorizations"`
}

func (c *client) Authorizations(ctx context.Context, ownerID string, reqOpts *twitchhttp.ReqOpts) (*AuthorizationsResponse, error) {
	if ownerID == "" {
		return nil, &twitchhttp.Error{
			StatusCode: 400,
			Message:    "No owner ID specified",
		}
	}

	query := url.Values{"owner_id": {ownerID}}
	req, err := c.NewRequest("GET", "/authorizations?"+query.Encode(), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.owl.authorizations",
		StatSampleRate: defaultStatSampleRate,
	})

	var data *AuthorizationsResponse
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

type validateSecretRequest struct {
	ClientSecret string `json:"client_secret"`
}

func (c *client) ValidateSecret(ctx context.Context, clientID, clientSecret string, reqOpts *twitchhttp.ReqOpts) (bool, error) {
	if clientID == "" {
		return false, ErrInvalidClientID
	}

	payload := validateSecretRequest{ClientSecret: clientSecret}
	body, err := json.Marshal(payload)
	if err != nil {
		return false, err
	}
	req, err := c.NewRequest("POST", fmt.Sprintf("/clients/%s/validate_secret", clientID), bytes.NewBuffer(body))
	if err != nil {
		return false, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.owl.validate_secret",
		StatSampleRate: defaultStatSampleRate,
	})

	res, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return false, err
	}

	return res.StatusCode == http.StatusNoContent, nil
}

type TokenResponse struct {
	OIDCRequired bool     `json:"oidc_required,omitempty"`
	Nonce        string   `json:"nonce,omitempty"`
	AccessToken  string   `json:"access_token"`
	ExpiresIn    int      `json:"expires_in"`
	RefreshToken string   `json:"refresh_token"`
	Scope        []string `json:"scope"`
}

func (c *client) ExchangeAuthorizationCode(ctx context.Context, clientID, clientSecret, code, redirectURI string, reqOpts *twitchhttp.ReqOpts) (*TokenResponse, error) {
	query := url.Values{
		"client_id":     {clientID},
		"client_secret": {clientSecret},
		"code":          {code},
		"redirect_uri":  {redirectURI},
		"grant_type":    {"authorization_code"},
	}

	req, err := c.NewRequest("POST", fmt.Sprintf("/token?%s", query.Encode()), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.owl.validate_secret",
		StatSampleRate: defaultStatSampleRate,
	})

	var response *TokenResponse
	_, err = c.DoJSON(ctx, &response, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return response, nil
}

type createAuthorizationRequest struct {
	OwnerID          string   `json:"owner_id"`
	ClientIdentifier string   `json:"client_id"`
	Scope            []string `json:"scope"`
}

func (c *client) CreateAuthorization(ctx context.Context, ownerID, clientID string, scopes []string, reqOpts *twitchhttp.ReqOpts) (*TokenResponse, error) {
	body, err := json.Marshal(createAuthorizationRequest{
		OwnerID:          ownerID,
		ClientIdentifier: clientID,
		Scope:            scopes,
	})
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest("POST", "/authorizations", bytes.NewBuffer(body))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.owl.create_authorization",
		StatSampleRate: defaultStatSampleRate,
	})

	var res *TokenResponse
	_, err = c.DoJSON(ctx, &res, req, combinedReqOpts)
	return res, err
}

func (c *client) DeleteAllSessions(ctx context.Context, ownerID string, reqOpts *twitchhttp.ReqOpts) error {
	req, err := c.NewRequest("DELETE", fmt.Sprintf("/v2/sessions/%s", ownerID), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.owl.delete_all_sessions",
		StatSampleRate: defaultStatSampleRate,
	})

	response, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	if response.StatusCode != http.StatusNoContent {
		return twitchhttp.HandleFailedResponse(response)
	}
	return nil
}
