package sessions

import (
	"errors"
	"fmt"
	"net"
	"strings"
	"time"

	"github.com/satori/go.uuid"

	"code.justin.tv/web/owl/tokengen"
)

const sessionTokenLength = 30

var (
	ErrInvalidToken = errors.New("Invalid token")
)

// Session represents an authorization of a client and associated metadata about the creation of the session
type Session struct {
	SessionID       string     `json:"-"`
	AuthorizationID string     `json:"-"`
	Scopes          []string   `json:"-"`
	UserinfoClaims  []string   `json:"-"`
	ClientRowID     string     `json:"-"`
	ClientID        string     `json:"client_id"`
	UserID          string     `json:"user_id"`
	CreatorID       string     `json:"creator_id"`
	ExpiresAt       *time.Time `json:"expires_at"`
	CreatedAt       *time.Time `json:"created_at"`
	UpdatedAt       *time.Time `json:"updated_at"`
	DeletedAt       *time.Time `json:"deleted_at"`
	Token           string     `json:"-"`
	Sequence        string     `json:"-"`
	Metadata        Metadata   `json:"metadata"`
}

// Metadata contains data about the creation of a session, such as the associated user agent
type Metadata struct {
	Location  string `json:"location"`
	IP        net.IP `json:"ip"`
	UserAgent string `json:"user_agent"`
}

// New creates a new session
func New(clientID, userID, creatorID string, expiresAt *time.Time, metadata Metadata) (*Session, error) {
	token, err := tokengen.RandomToken(sessionTokenLength)
	if err != nil {
		return nil, err
	}
	sequence, err := tokengen.RandomToken(sessionTokenLength)
	if err != nil {
		return nil, err
	}

	return &Session{
		SessionID: uuid.NewV4().String(),
		ClientID:  clientID,
		UserID:    userID,
		CreatorID: creatorID,
		ExpiresAt: expiresAt,
		Token:     token,
		Sequence:  sequence,
		Metadata:  metadata,
	}, nil

}

func (s *Session) Session() string {
	return fmt.Sprintf("%s:%s:%s", s.UserID, s.Token, s.Sequence)
}

func ParseToken(token string) (*Session, error) {
	parts := strings.Split(token, ":")
	if len(parts) != 3 {
		return nil, ErrInvalidToken
	}

	return &Session{
		UserID:   parts[0],
		Token:    parts[1],
		Sequence: parts[2],
	}, nil
}
