package partnerships

import (
	"encoding/json"
	"fmt"
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/web/partnerships/models"
)

const (
	defaultStatSampleRate = 1.0
	defaultTimingXactName = "partnerships_service"
)

// Client is an interface that exposes methods to fetch data from the partnerships service.
type Client interface {
	// GetProperties returns all partner properties of a partner
	GetPropertiesByUserID(ctx context.Context, userID int, reqOpts *twitchhttp.ReqOpts) (*models.PartnerProperties, error)
	// CheckPartneredByUserID returns a bool indicating if the user is partnered or not
	CheckPartneredByUserID(ctx context.Context, userID int, reqOpts *twitchhttp.ReqOpts) (bool, error)
}

type clientImpl struct {
	twitchhttp.Client
}

// NewClient creates a client for the partnerships service.
func NewClient(conf twitchhttp.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}

	twitchClient, err := twitchhttp.NewClient(conf)
	if err != nil {
		return nil, err
	}

	return &clientImpl{twitchClient}, nil
}

func (c *clientImpl) GetPropertiesByUserID(ctx context.Context, userID int, reqOpts *twitchhttp.ReqOpts) (*models.PartnerProperties, error) {
	path := fmt.Sprintf("/partners/%d/properties", userID)
	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.partnerships_service.get_properties",
		StatSampleRate: defaultStatSampleRate,
	})
	httpResp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}
	defer func() {
		err = httpResp.Body.Close()
	}()

	switch httpResp.StatusCode {
	case http.StatusOK:
		var decoded models.PartnerProperties
		if err := json.NewDecoder(httpResp.Body).Decode(&decoded); err != nil {
			return nil, err
		}
		return &decoded, nil

	default:
		// Unexpected result
		return nil, fmt.Errorf("Unexpected status code %d", httpResp.StatusCode)
	}
}

func (c *clientImpl) CheckPartneredByUserID(ctx context.Context, userID int, reqOpts *twitchhttp.ReqOpts) (bool, error) {
	properties, err := c.GetPropertiesByUserID(ctx, userID, reqOpts)
	if err != nil {
		return false, err
	}

	if properties.PartnerProgram == nil {
		return false, nil
	}

	return *properties.PartnerProgram, nil
}
