package api

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	. "github.com/stretchr/testify/mock"

	"time"

	auth "code.justin.tv/web/users-service/internal/auth/mocks"
	"code.justin.tv/web/users-service/logic/mocks"
	"code.justin.tv/web/users-service/models"
)

func TestGetBannedUsers(t *testing.T) {
	Convey("with a running api", t, func() {
		l := &mocks.Logic{}
		decoder := &auth.Decoder{}
		api, err := NewServer(l, decoder)
		So(err, ShouldBeNil)

		s := httptest.NewServer(api)
		defer s.Close()

		Convey("when /banned_users/?until=<time> is requested", func() {

			Convey("where time is valid and user id is string", func() {
				btime, err := time.Parse(time.RFC3339, "2016-09-17T11:33:18-08:00")
				if err != nil {
					t.Errorf("Error processing test time value %v\n", err)
				}
				id := "45573555"
				login := "fluffyfatcat"
				properties := []models.Properties{{ID: id, Login: &login}}
				l.On("GetBannedUsers", Anything, btime).Return(properties, nil)

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/banned_users?return_id_as_string=true", s.URL), nil)
				q := req.URL.Query()
				q.Add("until", btime.Format(time.RFC3339))
				req.URL.RawQuery = q.Encode()

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				So(resp.Status, ShouldEqual, "200 OK")

				var propertiesReturned struct {
					Results []models.Properties
				}

				err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
				So(err, ShouldBeNil)

				have := propertiesReturned.Results
				want := properties

				So(have, ShouldResemble, want)
			})
			Convey("where time is valid and user id is integer", func() {
				btime, err := time.Parse(time.RFC3339, "2016-09-17T11:33:18-08:00")
				if err != nil {
					t.Errorf("Error processing test time value %v\n", err)
				}
				sid := "45573555"
				id := uint64(45573555)
				login := "fluffyfatcat"
				properties := []models.Properties{{ID: sid, Login: &login}}
				intIDProperties := []models.IntIDProperties{{ID: id, Login: &login}}
				l.On("GetBannedUsers", Anything, btime).Return(properties, nil)

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/banned_users", s.URL), nil)
				q := req.URL.Query()
				q.Add("until", btime.Format(time.RFC3339))
				req.URL.RawQuery = q.Encode()

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				So(resp.Status, ShouldEqual, "200 OK")

				var propertiesReturned struct {
					Results []models.IntIDProperties
				}

				err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
				So(err, ShouldBeNil)

				have := propertiesReturned.Results
				want := intIDProperties

				So(have, ShouldResemble, want)
			})
			Convey("with an invalid time", func() {

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/banned_users?until=garbage&return_id_as_string=true", s.URL), nil)
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				Convey("the response has the correct HTTP status code", func() {
					So(resp.Status, ShouldEqual, "400 Bad Request")
				})
			})
		})
	})

}
