package api

import (
	"net/http"
	"net/url"

	"code.justin.tv/web/users-service/models"

	"goji.io/pat"
	"golang.org/x/net/context"
)

var ErrAdminLoginRequired = &models.CodedError{
	ErrorValue:      "An admin login must be provided for deletes",
	CodeValue:       "admin_login_required",
	StatusCodeValue: http.StatusBadRequest,
}

// unbanUser will unban a given user as specified by the ID parameter in the route.
func (S *Server) unbanUser(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	IDParam := pat.Param(r, "id")
	if !isValidID(IDParam) {
		return nil, models.ErrBadIdentifiers
	}

	ifResetCount := false
	if values.Get("reset_violation_counts") == "true" {
		ifResetCount = true
	}

	err := S.logic.UnbanUser(ctx, IDParam, ifResetCount)
	if err != nil {
		return nil, err
	}

	return nil, nil
}

// expireUserCache will expire a user's cached data.
func (S *Server) expireUserCache(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	IDParam := pat.Param(r, "id")
	if !isValidID(IDParam) {
		return nil, models.ErrBadIdentifiers
	}

	err := S.logic.OverwriteUserCache(ctx, IDParam)
	if err != nil {
		return nil, err
	}

	return nil, nil
}

func (S *Server) deleteUser(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	IDParam := pat.Param(r, "id")
	if !isValidID(IDParam) {
		return nil, models.ErrBadIdentifiers
	}

	adminLogin := values.Get("admin")
	if adminLogin == "" {
		return nil, ErrAdminLoginRequired
	}

	hardDelete := values.Get("destroy") == "true"
	if hardDelete {

		return nil, S.logic.HardDeleteUser(ctx, IDParam, adminLogin, values.Get("skip_block") == "true")
	}

	return nil, S.logic.SoftDeleteUser(ctx, IDParam, adminLogin)
}

func (S *Server) undeleteUser(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	IDParam := pat.Param(r, "id")
	if !isValidID(IDParam) {
		return nil, models.ErrBadIdentifiers
	}

	adminLogin := values.Get("admin")
	if adminLogin == "" {
		return nil, ErrAdminLoginRequired
	}

	return nil, S.logic.UndeleteUser(ctx, IDParam, adminLogin)
}
