package api_test

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/web/users-service/api"
	auth "code.justin.tv/web/users-service/internal/auth/mocks"
	"code.justin.tv/web/users-service/logic/mocks"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/assert"
	. "github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type DeleteTest struct {
	suite.Suite
	l       *mocks.Logic
	api     *api.Server
	s       *httptest.Server
	decoder *auth.Decoder
}

func (suite *DeleteTest) SetupTest() {
	var err error
	suite.l = &mocks.Logic{}
	suite.decoder = &auth.Decoder{}
	suite.api, err = api.NewServer(suite.l, suite.decoder)
	assert.NoError(suite.T(), err)
	suite.s = httptest.NewServer(suite.api)

	assert.NoError(suite.T(), err)
}

func (suite *DeleteTest) TestUnban() {
	Convey("when DELETE /users/:id/ban is requested with a bad ID", suite.T(), func() {
		invalidID := "garbagepete"
		req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/users/%v/ban", suite.s.URL, invalidID), nil)
		So(err, ShouldBeNil)

		resp, err := http.DefaultClient.Do(req)
		So(err, ShouldBeNil)

		Convey("the response has HTTP status 400 Bad Request", func() {
			So(resp.Status, ShouldEqual, "400 Bad Request")
		})
	})

	Convey("when DELETE /users/:id/ban is requested with a valid ID", suite.T(), func() {
		validID := "12"
		suite.l.On("UnbanUser", Anything, validID, false).Return(nil)
		defer suite.l.AssertExpectations(suite.T())
		req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/users/%s/ban", suite.s.URL, validID), nil)
		So(err, ShouldBeNil)

		resp, err := http.DefaultClient.Do(req)
		So(err, ShouldBeNil)
		Convey("the response has HTTP status 204 No Content", func() {
			So(resp.Status, ShouldEqual, "204 No Content")
		})
	})

	Convey("when DELETE /users/:id/ban is requested with a valid ID and reset_violation_counts", suite.T(), func() {
		validID := "12"
		suite.l.On("UnbanUser", Anything, validID, true).Return(nil)
		defer suite.l.AssertExpectations(suite.T())
		req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/users/%s/ban?reset_violation_counts=true", suite.s.URL, validID), nil)
		So(err, ShouldBeNil)

		resp, err := http.DefaultClient.Do(req)
		So(err, ShouldBeNil)
		Convey("the response has HTTP status 204 No Content", func() {
			So(resp.Status, ShouldEqual, "204 No Content")
		})
	})
}

func (suite *DeleteTest) TestDeleteUser() {
	Convey("when DELETE /users/:id is requested with bad ID", suite.T(), func() {
		invalidID := "garbagepete"
		req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/users/%v", suite.s.URL, invalidID), nil)
		So(err, ShouldBeNil)

		resp, err := http.DefaultClient.Do(req)
		So(err, ShouldBeNil)

		Convey("the response has HTTP status 400 Bad Request", func() {
			So(resp.Status, ShouldEqual, "400 Bad Request")
		})
	})

	Convey("when DELETE /users/:id is requested with valid ID and no admin", suite.T(), func() {
		validID := "12"
		req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/users/%v", suite.s.URL, validID), nil)
		So(err, ShouldBeNil)

		resp, err := http.DefaultClient.Do(req)
		So(err, ShouldBeNil)

		Convey("the response has HTTP status 400 Bad Request", func() {
			So(resp.Status, ShouldEqual, "400 Bad Request")
		})
	})

	Convey("when DELETE /users/:id is requested with valid ID and admin and destroy", suite.T(), func() {
		validID := "12"
		req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/users/%v?admin=test&destroy=true", suite.s.URL, validID), nil)
		So(err, ShouldBeNil)

		suite.l.On("HardDeleteUser", Anything, validID, "test", false).Return(nil)
		defer suite.l.AssertExpectations(suite.T())

		resp, err := http.DefaultClient.Do(req)
		So(err, ShouldBeNil)

		Convey("the response has HTTP status 204 No Content", func() {
			So(resp.Status, ShouldEqual, "204 No Content")
		})
		suite.l.AssertExpectations(suite.T())
	})

	Convey("when DELETE /users/:id is requested with valid ID and admin", suite.T(), func() {
		validID := "12"
		req, err := http.NewRequest("DELETE", fmt.Sprintf("%v/users/%v?admin=test", suite.s.URL, validID), nil)
		So(err, ShouldBeNil)

		suite.l.On("SoftDeleteUser", Anything, validID, "test").Return(nil)
		defer suite.l.AssertExpectations(suite.T())

		resp, err := http.DefaultClient.Do(req)
		So(err, ShouldBeNil)

		Convey("the response has HTTP status 204 No Content", func() {
			So(resp.Status, ShouldEqual, "204 No Content")
		})

	})
}

func (suite *DeleteTest) TestUndeleteUser() {
	Convey("when PATCH /users/:id/undelete is requested with bad ID", suite.T(), func() {
		invalidID := "garbagepete"
		req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%v/undelete", suite.s.URL, invalidID), nil)
		So(err, ShouldBeNil)

		resp, err := http.DefaultClient.Do(req)
		So(err, ShouldBeNil)

		Convey("the response has HTTP status 400 Bad Request", func() {
			So(resp.Status, ShouldEqual, "400 Bad Request")
		})
	})

	Convey("when PATCH /users/:id/undelete requested with valid ID and no admin", suite.T(), func() {
		validID := "12"
		req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%v/undelete", suite.s.URL, validID), nil)
		So(err, ShouldBeNil)

		resp, err := http.DefaultClient.Do(req)
		So(err, ShouldBeNil)

		Convey("the response has HTTP status 400 Bad Request", func() {
			So(resp.Status, ShouldEqual, "400 Bad Request")
		})
	})

	Convey("when PATCH /users/:id/undelete is requested with valid ID and admin", suite.T(), func() {
		validID := "12"
		req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%v/undelete?admin=test&destroy=true", suite.s.URL, validID), nil)
		So(err, ShouldBeNil)

		suite.l.On("UndeleteUser", Anything, validID, "test").Return(nil)
		defer suite.l.AssertExpectations(suite.T())

		resp, err := http.DefaultClient.Do(req)
		So(err, ShouldBeNil)

		Convey("the response has HTTP status 204 No Content", func() {
			So(resp.Status, ShouldEqual, "204 No Content")
		})
	})
}

func TestDeleteSuite(t *testing.T) {
	suite.Run(t, new(DeleteTest))
}
