package api

import (
	"fmt"
	"net/http"
	"net/url"
	"regexp"
	"strings"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/identity/golibs/middleware"
	"code.justin.tv/web/users-service/models"
	"goji.io/pat"
	"golang.org/x/net/context"
)

type UserPropertiesResult struct {
	Results        []models.Properties       `json:"results"`
	BadIdentifiers []models.ErrBadIdentifier `json:"bad_identifiers"`
}

type LoginTypePropertiesResult struct {
	Results []models.LoginTypeProperties `json:"results"`
}

func (u UserPropertiesResult) ConvertToIntIDPropertiesResult() (*UserIntIDPropertiesResult, error) {
	properties := []models.IntIDProperties{}
	for i := range u.Results {
		property, err := u.Results[i].ConvertToIntIDProperties()
		if err != nil {
			return nil, err
		}
		properties = append(properties, property)
	}
	return &UserIntIDPropertiesResult{properties}, nil
}

type UserIntIDPropertiesResult struct {
	Results []models.IntIDProperties `json:"results"`
}

func handleValidationErrorFunc(skipValidation, inlineValidation bool) func(err *models.ErrBadIdentifier, errs *[]models.ErrBadIdentifier) (bool, error) {
	return func(err *models.ErrBadIdentifier, errs *[]models.ErrBadIdentifier) (bool, error) {
		switch {
		case skipValidation:
			// Do nothing
		case inlineValidation:
			*errs = append(*errs, *err)
		default:
			return true, err
		}
		return false, nil
	}
}

func (S *Server) getProperties(ctx context.Context, values url.Values, r *http.Request) (respValue interface{}, respErr error) {

	IDs := values["id"]
	logins := values["login"]
	emails := values["email"]
	displaynames := values["displayname"]
	ips := values["ip"]
	loginLikes := values["login_like"]

	notDeleted := values.Get(models.NotDeletedParam) == "true"
	noTOS := values.Get(models.NoTOSViolationParam) == "true"
	noDMCA := values.Get(models.NoDMCAViolationParam) == "true"

	skipIdentiferValidation := values.Get("skip_identifier_validation") == "true"
	inlineIdentifierValidation := values.Get(models.InlineIdentifierValidation) == "true"

	response := &UserPropertiesResult{}

	validationErrorHandler := handleValidationErrorFunc(skipIdentiferValidation, inlineIdentifierValidation)

	// Customizes how errors are handled and returned. This function is handy for either collecting errors
	// to return in the response or exiting early when an error is encountered.
	handleValidationError := func(err *models.ErrBadIdentifier) bool {
		var ok bool
		ok, respErr = validationErrorHandler(err, &response.BadIdentifiers)
		return ok
	}

	var validatedIDs []string
	for _, id := range IDs {
		if !isValidID(id) {
			if handleValidationError(&models.ErrBadIdentifier{"id", id}) {
				return nil, models.ErrBadIdentifiers
			}
			continue
		}

		validatedIDs = append(validatedIDs, id)
	}

	var validatedLogins []string
	for _, login := range logins {
		if !isValidLogin(login) {
			if handleValidationError(&models.ErrBadIdentifier{"login", login}) {
				return
			}
			continue
		}
		validatedLogins = append(validatedLogins, strings.ToLower(login))
	}

	for eidx, email := range emails {
		emails[eidx] = strings.ToLower(email)
	}

	for didx, displayname := range displaynames {
		displaynames[didx] = strings.ToLower(displayname)
	}

	var validatedIPs []string
	for _, ip := range ips {
		m, err := regexp.MatchString("^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$", ip)
		if !m || err != nil {
			if handleValidationError(&models.ErrBadIdentifier{"ip", ip}) {
				return
			}
			continue
		}

		validatedIPs = append(validatedIPs, ip)
	}

	if len(loginLikes) == 0 && len(validatedLogins) == 0 && len(validatedIDs) == 0 && len(emails) == 0 && len(displaynames) == 0 && len(validatedIPs) == 0 {
		if (len(IDs)+len(logins)+len(ips)) > 0 && inlineIdentifierValidation {
			// When the user has actually provided identifiers, but they were filtered
			// because inline identifier validation was enabled, return an empty response.
			return response, nil
		}

		return nil, models.ErrNoIdentifiers
	}

	var err error

	if len(loginLikes) > 0 {
		// If multiple login_like are specified, use the first param only
		loginLike := strings.ToLower(strings.TrimSpace(loginLikes[0]))
		if loginLike == "" {
			return nil, &models.ErrBadIdentifier{"login_like", loginLike}
		}
		response.Results, err = S.logic.GetUserPropertiesLike(ctx, "login", loginLike)
	} else {
		params := &models.FilterParams{
			IDs:          validatedIDs,
			Logins:       validatedLogins,
			Emails:       emails,
			DisplayNames: displaynames,
			Ips:          validatedIPs,

			NotDeleted:      notDeleted,
			NoTOSViolation:  noTOS,
			NoDMCAViolation: noDMCA,
		}
		response.Results, err = S.logic.GetUserPropertiesBulk(ctx, params)
	}

	if err != nil {
		return nil, err
	}

	asID := values.Get("return_id_as_string") == "true"
	reportAsID(ctx, "bulk_get_users", asID)
	if !asID {
		return response.ConvertToIntIDPropertiesResult()
	}
	return response, nil

}

func (S *Server) getRenameEligiblity(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	id := pat.Param(r, "id")

	if !isValidID(id) {
		return nil, models.ErrBadIdentifiers
	}

	upresult, err := S.logic.GetRenameEligibility(ctx, id)
	if err != nil {
		return nil, err
	}

	return &upresult, nil
}

func (S *Server) getLoginType(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	logins := values["login"]

	for lidx, login := range logins {
		if !isValidLogin(login) {
			return nil, &models.ErrBadIdentifier{"login", login}
		}
		logins[lidx] = strings.ToLower(login)
	}

	if len(logins) == 0 {
		return nil, models.ErrNoIdentifiers
	}

	if len(logins) > 100 {
		return nil, models.ErrTooManyIdentifiers
	}

	requestingUserID := values.Get("requesting_user_id")
	isRename := values.Get("is_rename") == "true"

	lpresults, _, err := S.logic.GetTypeForLogins(ctx, logins, requestingUserID, isRename, false)
	if err != nil {
		return nil, err
	}
	return &LoginTypePropertiesResult{Results: lpresults}, nil
}

func (S *Server) getUser(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	id := pat.Param(r, "id")

	if !isValidID(id) {
		return nil, models.ErrBadIdentifiers
	}

	notDeleted := values.Get(models.NotDeletedParam) == "true"
	noTOS := values.Get(models.NoTOSViolationParam) == "true"
	noDMCA := values.Get(models.NoDMCAViolationParam) == "true"

	params := &models.FilterParams{
		NotDeleted:      notDeleted,
		NoTOSViolation:  noTOS,
		NoDMCAViolation: noDMCA,
	}

	upresult, err := S.logic.GetUserPropertiesByID(ctx, id, params)
	if err != nil {
		return nil, err
	}

	asID := values.Get("return_id_as_string") == "true"
	reportAsID(ctx, "get_user", asID)
	if !asID {
		upintresult, err := upresult.ConvertToIntIDProperties()
		if err != nil {
			return nil, err
		}
		return &upintresult, nil
	}

	return &upresult, nil
}

func reportAsID(ctx context.Context, name string, asID bool) {
	stats := config.ClusterStatsd()
	source := middleware.GetSource(ctx)
	// Ensure the source name groups into one subfolder
	source = strings.Replace(source, ".", "_", -1)
	metric := fmt.Sprintf("return_id_as_string.%s.%s.%t", source, name, asID)
	if err := stats.Inc(metric, 1, 0.1); err != nil {
		logx.Warn(context.Background(), fmt.Sprintf("failed to report stats counter for %q", metric))
	}
}
