package api

import (
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"golang.org/x/net/context"

	"code.justin.tv/web/users-service/models"
)

func processChannelPropertiesResults(cpresults []models.ChannelProperties) []models.ChannelProperties {
	for i, result := range cpresults {
		if result.Game != nil && strings.ToLower(*(result.Game)) == "the playroom" {
			cpresults[i].DirectoryHidden = true
		}
	}
	return cpresults
}

func (S *Server) getChannelProperties(ctx context.Context, values url.Values, r *http.Request) (respVal interface{}, respErr error) {
	ids := values["id"]
	names := values["name"]

	var validIDs []uint64
	var validNames []string

	response := &models.ChannelPropertiesResult{}

	inlineIdentifierValidation := values.Get(models.InlineIdentifierValidation) == "true"

	validationErrorHandler := handleValidationErrorFunc(false, inlineIdentifierValidation)

	// Customizes how errors are handled and returned. This function is handy for either collecting errors
	// to return in the response or exiting early when an error is encountered.
	handleValidationError := func(err *models.ErrBadIdentifier) bool {
		var ok bool
		ok, respErr = validationErrorHandler(err, &response.BadIdentifiers)
		return ok
	}

	for _, id := range ids {
		j, err := strconv.Atoi(id)
		if err != nil || !isValidID(id) {
			if handleValidationError(&models.ErrBadIdentifier{"id", id}) {
				return nil, models.ErrBadChannelIdentifiers
			}
			continue
		}
		validIDs = append(validIDs, uint64(j))
	}

	for _, name := range names {
		if !isValidLogin(name) {
			if handleValidationError(&models.ErrBadIdentifier{"name", name}) {
				return nil, models.ErrBadChannelIdentifiers
			}
			continue
		}
		validNames = append(validNames, strings.ToLower(name))
	}

	if len(validIDs) == 0 && len(validNames) == 0 {
		if (len(ids)+len(names) > 0) && inlineIdentifierValidation {
			return response, nil
		}
		return nil, models.ErrNoChannelIdentifiers
	}

	notDeleted := values.Get(models.NotDeletedParam) == "true"
	noTOS := values.Get(models.NoTOSViolationParam) == "true"
	noDMCA := values.Get(models.NoDMCAViolationParam) == "true"
	params := &models.ChannelFilterParams{
		NotDeleted:      notDeleted,
		NoTOSViolation:  noTOS,
		NoDMCAViolation: noDMCA,
	}

	cps := []models.ChannelProperties{}

	cps, err := S.logic.GetAllChannelPropertiesBulk(ctx, validIDs, validNames, params)
	if err != nil {
		return nil, err
	}

	response.Results = processChannelPropertiesResults(cps)

	return response, nil
}
