package api

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	. "github.com/stretchr/testify/mock"

	"code.justin.tv/web/users-service/backend/channels"
	auth "code.justin.tv/web/users-service/internal/auth/mocks"
	"code.justin.tv/web/users-service/logic/mocks"
	"code.justin.tv/web/users-service/models"
)

func TestGetProperties(t *testing.T) {

	Convey("with a running api", t, func() {
		cpr := &mocks.Logic{}
		decoder := &auth.Decoder{}
		api, err := NewServer(cpr, decoder)
		So(err, ShouldBeNil)

		s := httptest.NewServer(api)
		defer s.Close()

		Convey("when /channels is requested", func() {
			Convey("with a valid name", func() {
				name := "creative"
				properties := []models.ChannelProperties{{Name: name}}
				cpr.On("GetAllChannelPropertiesBulk", Anything, []uint64(nil), []string{name}, &models.ChannelFilterParams{}).Return(properties, nil)

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/channels?name=%s", s.URL, name), nil)
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				Convey("the response has the correct HTTP status code", func() {
					So(resp.Status, ShouldEqual, "200 OK")
				})

				Convey("the response body contains a JSON encoded representation of the properties", func() {
					var propertiesReturned *models.ChannelPropertiesResult
					err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
					So(err, ShouldBeNil)
					So(len(propertiesReturned.Results), ShouldEqual, 1)
					So(len(propertiesReturned.BadIdentifiers), ShouldEqual, 0)
					So(propertiesReturned.Results, ShouldResemble, properties)
				})
			})

			Convey("with a name that has no properties", func() {
				name := "daikatana"
				cpr.On("GetAllChannelPropertiesBulk", Anything, []uint64(nil), []string{name}, &models.ChannelFilterParams{}).Return(nil, channels.ErrNoChannelProperties)

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/channels?name=%s", s.URL, name), nil)
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				Convey("the response has the correct HTTP status code", func() {
					So(resp.Status, ShouldEqual, "404 Not Found")
				})
			})

			Convey("with a valid id", func() {
				ID := uint64(5)
				properties := []models.ChannelProperties{{ID: ID}}
				cpr.On("GetAllChannelPropertiesBulk", Anything, []uint64{ID}, []string(nil), &models.ChannelFilterParams{}).Return(properties, nil)

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/channels?id=%d", s.URL, ID), nil)
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				Convey("the response has the correct HTTP status code", func() {
					So(resp.Status, ShouldEqual, "200 OK")
				})

				Convey("the response body contains a JSON encoded representation of the properties", func() {
					var propertiesReturned *models.ChannelPropertiesResult
					err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
					So(err, ShouldBeNil)
					So(len(propertiesReturned.Results), ShouldEqual, 1)
					So(len(propertiesReturned.BadIdentifiers), ShouldEqual, 0)
					So(propertiesReturned.Results, ShouldResemble, properties)
				})
			})

			Convey("with an invalid,valid id and inline identifier validation", func() {
				invalidID := "garbage"
				validName := "creative"

				name := "creative"
				properties := []models.ChannelProperties{{Name: name}}
				cpr.On("GetAllChannelPropertiesBulk", Anything, []uint64(nil), []string{name}, &models.ChannelFilterParams{}).Return(properties, nil)

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/channels?id=%s&name=%s&inline_identifier_validation=true", s.URL, invalidID, validName), nil)
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				Convey("the response has the correct HTTP status code", func() {
					So(resp.Status, ShouldEqual, "200 OK")
				})

				Convey("the response body contains a JSON encoded representation of the properties", func() {
					var propertiesReturned *models.ChannelPropertiesResult
					err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
					So(err, ShouldBeNil)
					So(len(propertiesReturned.Results), ShouldEqual, 1)
					So(len(propertiesReturned.BadIdentifiers), ShouldEqual, 1)
					So(propertiesReturned.Results, ShouldResemble, properties)
				})
			})

			Convey("with an invalid id that is larger than maxID", func() {
				invalidID := maxID + 2

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/channels?id=%d", s.URL, invalidID), nil)
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				Convey("the response has the correct HTTP status code", func() {
					So(resp.Status, ShouldEqual, "400 Bad Request")
				})
			})

			Convey("with an invalid id and inline parameters", func() {
				ID := "garbage"

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/channels?id=%s", s.URL, ID), nil)
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				Convey("the response has the correct HTTP status code", func() {
					So(resp.Status, ShouldEqual, "400 Bad Request")
				})
			})

			Convey("with an invalid id and inline validation", func() {
				ID := "garbage"

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/channels?id=%s&inline_identifier_validation=true", s.URL, ID), nil)
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				Convey("the response has the correct HTTP status code", func() {
					So(resp.Status, ShouldEqual, "200 OK")
				})
			})

			Convey("with an ID that has no properties", func() {
				ID := uint64(444)
				cpr.On("GetAllChannelPropertiesBulk", Anything, []uint64{ID}, []string(nil), &models.ChannelFilterParams{}).Return(nil, channels.ErrNoChannelProperties)

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/channels?id=%d", s.URL, ID), nil)
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				Convey("the response has the correct HTTP status code", func() {
					So(resp.Status, ShouldEqual, "404 Not Found")
				})
			})

			Convey("without a name or id parameter", func() {
				name := "creative"
				cpr.On("GetAllChannelPropertiesBulk", Anything, []uint64(nil), name, &models.ChannelFilterParams{}).Return(nil, models.ErrNoChannelIdentifiers)

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/channels?blah=%s", s.URL, name), nil)
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				Convey("the response has the correct HTTP status code", func() {
					So(resp.Status, ShouldEqual, "400 Bad Request")
				})
			})

			Convey("when the backend is dead", func() {
				name := "creative"
				id := uint64(5)

				cpr.On("GetAllChannelPropertiesBulk", Anything, []uint64(nil), []string{name}, &models.ChannelFilterParams{}).Return(nil, errors.New("Nope"))
				cpr.On("GetAllChannelPropertiesBulk", Anything, []uint64{id}, []string(nil), &models.ChannelFilterParams{}).Return(nil, errors.New("Nope"))
				cpr.On("GetAllChannelPropertiesBulk", Anything, []uint64{id}, []string{name}, &models.ChannelFilterParams{}).Return(nil, errors.New("Nope"))

				Convey("and a name is requested", func() {

					req, err := http.NewRequest("GET", fmt.Sprintf("%v/channels?name=%s", s.URL, name), nil)
					So(err, ShouldBeNil)

					resp, err := http.DefaultClient.Do(req)
					So(err, ShouldBeNil)

					Convey("the response has the correct HTTP status code", func() {
						So(resp.Status, ShouldEqual, "500 Internal Server Error")
					})
				})
				Convey("and an ID is requested", func() {

					req, err := http.NewRequest("GET", fmt.Sprintf("%v/channels?id=%d", s.URL, id), nil)
					So(err, ShouldBeNil)

					resp, err := http.DefaultClient.Do(req)
					So(err, ShouldBeNil)

					Convey("the response has the correct HTTP status code", func() {
						So(resp.Status, ShouldEqual, "500 Internal Server Error")
					})
				})
				Convey("and both a name and an ID are requested", func() {

					req, err := http.NewRequest("GET", fmt.Sprintf("%v/channels?id=%d&name=%s", s.URL, id, name), nil)
					So(err, ShouldBeNil)

					resp, err := http.DefaultClient.Do(req)
					So(err, ShouldBeNil)

					Convey("the response has the correct HTTP status code", func() {
						So(resp.Status, ShouldEqual, "500 Internal Server Error")
					})
				})
			})

		}) //End of "when /channels is requested"
	})
}
