package api_test

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"encoding/json"
	"errors"

	"code.justin.tv/web/users-service/api"
	authmocks "code.justin.tv/web/users-service/internal/auth/mocks"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/logic/mocks"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type GetExternalTest struct {
	suite.Suite
	l       *mocks.Logic
	api     *api.Server
	s       *httptest.Server
	decoder *authmocks.Decoder
}

func (suite *GetExternalTest) SetupTest() {
	var err error
	suite.l = &mocks.Logic{}
	suite.decoder = &authmocks.Decoder{}
	suite.api, err = api.NewServer(suite.l, suite.decoder)
	assert.NoError(suite.T(), err)
	suite.s = httptest.NewServer(suite.api)
}

func (suite *GetExternalTest) TestGerUserExternalSuccess() {
	suite.l.On("GetUserPropertiesByID", mock.Anything, AdminUserID, mock.Anything).Return(Users[AdminUserID], nil)
	suite.decoder.On("AuthorizeUserRead", mock.Anything, mock.Anything, AdminUserID).Return(errors.New("Auth failed"))

	req, err := http.NewRequest("GET", fmt.Sprintf("%v/users/%s/external?requester=%s", suite.s.URL, AdminUserID, AdminUserID), nil)
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)

	var propertiesReturned *models.Properties
	err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
	assert.NoError(suite.T(), err)

	assert.Equal(suite.T(), resp.StatusCode, 200)
	assert.Equal(suite.T(), *propertiesReturned, Users[AdminUserID].ConvertToExternal())

	suite.l.AssertExpectations(suite.T())
}

func (suite *GetExternalTest) TestGerUserExternalWithAuthSuccess() {
	suite.l.On("GetUserPropertiesByID", mock.Anything, AdminUserID, mock.Anything).Return(Users[AdminUserID], nil)
	suite.decoder.On("AuthorizeUserRead", mock.Anything, mock.Anything, AdminUserID).Return(nil)

	req, err := http.NewRequest("GET", fmt.Sprintf("%v/users/%s/external?requester=%s", suite.s.URL, AdminUserID, AdminUserID), nil)
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)

	var propertiesReturned models.Properties
	err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 200)
	assert.Equal(suite.T(), propertiesReturned, *Users[AdminUserID])

	suite.l.AssertExpectations(suite.T())
}

func (suite *GetExternalTest) TestGerUserExternalWithoutAuthSuccess() {
	suite.l.On("GetUserPropertiesByID", mock.Anything, AdminUserID, mock.Anything).Return(Users[AdminUserID], nil)
	suite.decoder.On("AuthorizeUserRead", mock.Anything, mock.Anything, AdminUserID).Return(errors.New("Auth failed"))

	req, err := http.NewRequest("GET", fmt.Sprintf("%v/users/%s/external?requester=%s", suite.s.URL, AdminUserID, AdminUserID), nil)
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)

	var propertiesReturned models.Properties
	err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 200)
	assert.Equal(suite.T(), propertiesReturned, Users[AdminUserID].ConvertToExternal())

	suite.l.AssertExpectations(suite.T())
}

func (suite *GetExternalTest) TestGerUserExternalWithoutEmptyRequester() {
	suite.l.On("GetUserPropertiesByID", mock.Anything, AdminUserID, mock.Anything).Return(Users[AdminUserID], nil)
	suite.decoder.On("AuthorizeUserRead", mock.Anything, mock.Anything, "").Return(errors.New("Auth failed"))

	req, err := http.NewRequest("GET", fmt.Sprintf("%v/users/%s/external", suite.s.URL, AdminUserID), nil)
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)

	var propertiesReturned models.Properties
	err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 200)
	assert.Equal(suite.T(), propertiesReturned, Users[AdminUserID].ConvertToExternal())

	suite.l.AssertExpectations(suite.T())
}

func (suite *GetExternalTest) TestGetPropertiesExternalWithAuthSuccess() {
	pr := models.PropertiesResult{Results: []*models.Properties{Users[AdminUserID]}}
	suite.l.On("GetUserPropertiesBulk", mock.Anything, mock.Anything).Return([]models.Properties{*Users[AdminUserID]}, nil)
	suite.decoder.On("AuthorizeBulkUserRead", mock.Anything, mock.Anything, AdminUserID).Return(nil)

	req, err := http.NewRequest("GET", fmt.Sprintf("%v/users/external?id=%s&requester=%s", suite.s.URL, AdminUserID, AdminUserID), nil)
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)

	var propertiesReturned models.PropertiesResult
	err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 200)
	assert.Equal(suite.T(), propertiesReturned, pr)

	suite.l.AssertExpectations(suite.T())
}

func (suite *GetExternalTest) TestGetPropertiesExternalWithoutAuthSuccess() {
	pr := models.PropertiesResult{Results: []*models.Properties{Users[AdminUserID]}}
	suite.l.On("GetUserPropertiesBulk", mock.Anything, mock.Anything).Return([]models.Properties{*Users[AdminUserID]}, nil)
	suite.decoder.On("AuthorizeBulkUserRead", mock.Anything, mock.Anything, AdminUserID).Return(errors.New("Auth Failed"))

	req, err := http.NewRequest("GET", fmt.Sprintf("%v/users/external?id=%s&requester=%s", suite.s.URL, AdminUserID, AdminUserID), nil)
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)

	var propertiesReturned models.PropertiesResult
	err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 200)
	assert.Equal(suite.T(), propertiesReturned, pr.ConvertToExternal())

	suite.l.AssertExpectations(suite.T())
}

func (suite *GetExternalTest) TestGetPropertiesExternalWithEmptyRequester() {
	pr := models.PropertiesResult{Results: []*models.Properties{Users[AdminUserID]}}
	suite.l.On("GetUserPropertiesBulk", mock.Anything, mock.Anything).Return([]models.Properties{*Users[AdminUserID]}, nil)
	suite.decoder.On("AuthorizeBulkUserRead", mock.Anything, mock.Anything, "").Return(errors.New("Auth Failed"))

	req, err := http.NewRequest("GET", fmt.Sprintf("%v/users/external?id=%s", suite.s.URL, AdminUserID), nil)
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)

	var propertiesReturned models.PropertiesResult
	err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 200)
	assert.Equal(suite.T(), propertiesReturned, pr.ConvertToExternal())

	suite.l.AssertExpectations(suite.T())
}

func TestGetExternalSuite(t *testing.T) {
	suite.Run(t, new(GetExternalTest))
}
