package api

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	. "github.com/stretchr/testify/mock"

	"code.justin.tv/web/users-service/backend/util"
	auth "code.justin.tv/web/users-service/internal/auth/mocks"
	"code.justin.tv/web/users-service/logic/mocks"
	"code.justin.tv/web/users-service/models"
)

type TestUserResults struct {
	Results []TestUserProperties
}

type TestUserProperties struct {
	ID    string  `json:"id" `
	Login *string `json:"login" `
	Email *string `json:"email" `
}

func TestHandleProperties(t *testing.T) {
	Convey("with a running api", t, func() {
		l := &mocks.Logic{}
		decoder := &auth.Decoder{}

		api, err := NewServer(l, decoder)
		So(err, ShouldBeNil)

		s := httptest.NewServer(api)
		defer s.Close()

		Convey("when /users? is requested", func() {
			Convey("and login/email are mixed capitilzation we should lowercase them", func() {
				login := "GarGoyle"
				email := "BoB@GarBage.com"
				id := "45573555"
				ip := "192.168.1.123"
				properties := []models.Properties{{ID: id, Login: &login}}

				l.On("GetUserPropertiesBulk", Anything, &models.FilterParams{Logins: []string{strings.ToLower(login)}, Emails: []string{strings.ToLower(email)}, Ips: []string{ip}}).Return(properties, nil)

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/users?return_id_as_string=true", s.URL), nil)
				q := req.URL.Query()
				q.Add("login", login)
				q.Add("email", email)
				q.Add("ip", ip)

				req.URL.RawQuery = q.Encode()
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				l.AssertExpectations(t)

				So(resp.Status, ShouldEqual, "200 OK")
			})
		})

		Convey("when /users? is requested and people want integer as user id for return", func() {
			login := "GarGoyle"
			email := "bob@garbadge.com"
			sid := "45573555"
			id := uint64(45573555)
			properties := []models.Properties{{ID: sid, Login: &login}}
			intIDProperties := []models.IntIDProperties{{ID: id, Login: &login}}
			l.On("GetUserPropertiesBulk", Anything, &models.FilterParams{Logins: []string{strings.ToLower(login)}, Emails: []string{strings.ToLower(email)}}).Return(properties, nil)

			req, err := http.NewRequest("GET", fmt.Sprintf("%v/users", s.URL), nil)
			q := req.URL.Query()
			q.Add("login", login)
			q.Add("email", email)

			req.URL.RawQuery = q.Encode()
			So(err, ShouldBeNil)

			resp, err := http.DefaultClient.Do(req)
			So(err, ShouldBeNil)

			l.AssertExpectations(t)

			So(resp.Status, ShouldEqual, "200 OK")

			var propertiesReturned struct {
				Results []models.IntIDProperties
			}

			err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
			So(err, ShouldBeNil)

			have := propertiesReturned.Results
			want := intIDProperties

			So(have, ShouldResemble, want)
		})

		Convey("when /users?id=<id>&prop=<prop> is requested", func() {
			Convey("where prop={login}", func() {
				props := []string{"login"}
				Convey("with a valid ID", func() {
					id := "45573555"
					login := "fluffyfatcat"
					properties := []models.Properties{{ID: id, Login: &login}}
					l.On("GetUserPropertiesBulk", Anything, &models.FilterParams{IDs: []string{id}}).Return(properties, nil)

					req, err := http.NewRequest("GET", fmt.Sprintf("%v/users?return_id_as_string=true", s.URL), nil)
					q := req.URL.Query()
					q.Add("id", fmt.Sprintf("%v", id))
					for _, prop := range props {
						q.Add("prop", prop)
					}
					req.URL.RawQuery = q.Encode()
					So(err, ShouldBeNil)

					resp, err := http.DefaultClient.Do(req)
					So(err, ShouldBeNil)

					Convey("the response has the correct HTTP status code", func() {
						So(resp.Status, ShouldEqual, "200 OK")
					})

					Convey("the response body contains a JSON encoded representation of the properties", func() {

						var propertiesReturned TestUserResults

						err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
						So(err, ShouldBeNil)

						So(len(propertiesReturned.Results), ShouldEqual, 1)

						have := propertiesReturned.Results[0]
						want := TestUserProperties{ID: id, Login: &login}

						So(have, ShouldResemble, want)
					})
				})

				Convey("with an invalid ID", func() {
					id := "test"

					req, err := http.NewRequest("GET", fmt.Sprintf("%v/users?id=%s&prop=login?return_id_as_string=true", s.URL, id), nil)
					So(err, ShouldBeNil)

					resp, err := http.DefaultClient.Do(req)
					So(err, ShouldBeNil)

					Convey("the response has the correct HTTP status code", func() {
						So(resp.Status, ShouldEqual, "400 Bad Request")
					})
				})

				Convey("with an invalid, valid ID and inline validation", func() {
					invalidID := "test"
					validID := "1"

					l.On("GetUserPropertiesBulk", Anything, &models.FilterParams{IDs: []string{validID}}).Return([]models.Properties{{ID: validID}}, nil)

					req, err := http.NewRequest("GET", fmt.Sprintf("%v/users?id=%s&id=%s&prop=login&return_id_as_string=true&inline_identifier_validation=true", s.URL, invalidID, validID), nil)
					So(err, ShouldBeNil)

					resp, err := http.DefaultClient.Do(req)
					So(err, ShouldBeNil)

					Convey("the response has the correct HTTP status code", func() {
						So(resp.Status, ShouldEqual, "200 OK")
					})

					Convey("the response body contains a JSON encoded representation of the properties", func() {
						var propertiesReturned *models.PropertiesResult
						err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
						So(err, ShouldBeNil)
						So(len(propertiesReturned.Results), ShouldEqual, 1)
						So(len(propertiesReturned.BadIdentifiers), ShouldEqual, 1)
					})

					l.AssertExpectations(t)
				})

				Convey("with an invalid and inline validation", func() {
					invalidID := "test"

					req, err := http.NewRequest("GET", fmt.Sprintf("%v/users?id=%s&prop=login&return_id_as_string=true&inline_identifier_validation=true", s.URL, invalidID), nil)
					So(err, ShouldBeNil)

					resp, err := http.DefaultClient.Do(req)
					So(err, ShouldBeNil)

					Convey("the response has the correct HTTP status code", func() {
						So(resp.Status, ShouldEqual, "200 OK")
					})

					Convey("the response body contains a JSON encoded representation of the properties", func() {
						var propertiesReturned *models.PropertiesResult
						err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
						So(err, ShouldBeNil)
						So(len(propertiesReturned.BadIdentifiers), ShouldEqual, 1)
					})

					l.AssertExpectations(t)
				})

				Convey("with an ID that has no properties", func() {
					id := "42"
					l.On("GetUserPropertiesBulk", Anything, &models.FilterParams{IDs: []string{id}}).Return(nil, util.ErrNoProperties)

					req, err := http.NewRequest("GET", fmt.Sprintf("%v/users?id=%s&prop=login?return_id_as_string=true", s.URL, id), nil)
					So(err, ShouldBeNil)

					resp, err := http.DefaultClient.Do(req)
					So(err, ShouldBeNil)

					Convey("the response has the correct HTTP status code", func() {
						So(resp.Status, ShouldEqual, "404 Not Found")
					})
				})
			})
		})

		Convey("when /users?login=<login>&prop=<prop> is requested", func() {
			Convey("where prop={login}", func() {
				props := []string{"login"}
				Convey("with a valid login", func() {
					id := "32264361"
					login := "patches"

					properties := []models.Properties{{ID: id, Login: &login}}
					l.On("GetUserPropertiesBulk", Anything, &models.FilterParams{Logins: []string{login}}).Return(properties, nil)

					req, err := http.NewRequest("GET", fmt.Sprintf("%v/users?return_id_as_string=true", s.URL), nil)
					q := req.URL.Query()
					q.Add("login", fmt.Sprintf("%v", login))
					for _, prop := range props {
						q.Add("prop", prop)
					}
					req.URL.RawQuery = q.Encode()
					So(err, ShouldBeNil)

					resp, err := http.DefaultClient.Do(req)
					So(err, ShouldBeNil)

					Convey("the response has the correct HTTP status code", func() {
						So(resp.Status, ShouldEqual, "200 OK")
					})

					Convey("the response body contains a JSON encoded representation of the properties", func() {

						var propertiesReturned TestUserResults

						err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
						So(err, ShouldBeNil)

						So(len(propertiesReturned.Results), ShouldEqual, 1)

						have := propertiesReturned.Results[0]
						want := TestUserProperties{ID: id, Login: &login}

						So(have, ShouldResemble, want)
					})
				})
				Convey("with a login that has no properties", func() {
					login := "unkindled"
					l.On("GetUserPropertiesBulk", Anything, &models.FilterParams{Logins: []string{login}}).Return(nil, util.ErrNoProperties)

					req, err := http.NewRequest("GET", fmt.Sprintf("%v/users?login=%v&prop=login?return_id_as_string=true", s.URL, login), nil)
					So(err, ShouldBeNil)

					resp, err := http.DefaultClient.Do(req)
					So(err, ShouldBeNil)

					Convey("the response has the correct HTTP status code", func() {
						So(resp.Status, ShouldEqual, "404 Not Found")
					})
				})
			})
		})

		Convey("when /users?email=<email>&prop=<prop> is requested", func() {
			Convey("where prop={email}", func() {
				props := []string{"login"}
				Convey("with a valid email", func() {
					id := "32264361"
					email := "patches@twitch.tv"

					properties := []models.Properties{{ID: id, Email: &email}}
					l.On("GetUserPropertiesBulk", Anything, &models.FilterParams{Emails: []string{email}}).Return(properties, nil)

					req, err := http.NewRequest("GET", fmt.Sprintf("%v/users?return_id_as_string=true", s.URL), nil)
					q := req.URL.Query()
					q.Add("email", fmt.Sprintf("%v", email))
					for _, prop := range props {
						q.Add("prop", prop)
					}
					req.URL.RawQuery = q.Encode()
					So(err, ShouldBeNil)

					resp, err := http.DefaultClient.Do(req)
					So(err, ShouldBeNil)

					Convey("the response has the correct HTTP status code", func() {
						So(resp.Status, ShouldEqual, "200 OK")
					})

					Convey("the response body contains a JSON encoded representation of the properties", func() {

						var propertiesReturned TestUserResults

						err = json.NewDecoder(resp.Body).Decode(&propertiesReturned)
						So(err, ShouldBeNil)

						So(len(propertiesReturned.Results), ShouldEqual, 1)

						have := propertiesReturned.Results[0]
						want := TestUserProperties{ID: id, Email: &email}

						So(have, ShouldResemble, want)
					})
				})
				Convey("with a login that has no properties", func() {
					email := "unkindled@twitch.tv"
					l.On("GetUserPropertiesBulk", Anything, &models.FilterParams{Emails: []string{email}}).Return(nil, util.ErrNoProperties)

					req, err := http.NewRequest("GET", fmt.Sprintf("%v/users?email=%v&prop=login?return_id_as_string=true", s.URL, email), nil)
					So(err, ShouldBeNil)

					resp, err := http.DefaultClient.Do(req)
					So(err, ShouldBeNil)

					Convey("the response has the correct HTTP status code", func() {
						So(resp.Status, ShouldEqual, "404 Not Found")
					})
				})
			})
		})

		Convey("when /users/{id} is requested", func() {
			Convey("where {id} is invalid", func() {
				id := "asdf"

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/users/%s", s.URL, id), nil)
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				Convey("the response has the correct HTTP status code", func() {
					So(resp.StatusCode, ShouldEqual, 400)
				})
			})
			Convey("where {id} is valid", func() {
				user := &models.Properties{
					ID: "125",
				}

				l.On("GetUserPropertiesByID", Anything, user.ID, Anything).Return(user, nil)

				req, err := http.NewRequest("GET", fmt.Sprintf("%v/users/%s", s.URL, user.ID), nil)
				So(err, ShouldBeNil)

				resp, err := http.DefaultClient.Do(req)
				So(err, ShouldBeNil)

				Convey("the response has the correct HTTP status code", func() {
					So(resp.StatusCode, ShouldEqual, 200)
				})
			})
		})
	})
}
