package api

import (
	"net/http"
	"net/url"

	"fmt"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/chitin"
	"code.justin.tv/foundation/xray"
	"code.justin.tv/web/users-service/configs"
	"golang.org/x/net/context"
)

type apiFunc func(context.Context, url.Values, *http.Request) (interface{}, error)

// CreateHandler creates a goji.HandlerFunc with the apiFunc that is passed in
func CreateHandler(handler apiFunc) func(w http.ResponseWriter, r *http.Request) {
	return func(w http.ResponseWriter, r *http.Request) {
		ctx := r.Context()
		values := r.URL.Query()

		defer func() {
			// Do not use logx.RecoverAndLog(), recovers do not work in when not inline
			if err := recover(); err != nil {
				logx.Error(context.Background(), errx.New(fmt.Errorf("panic recovered: %v", err)))
			}
		}()

		ctx = logx.WithFields(ctx, logx.Fields{
			"method":            r.Method,
			"ip":                GetIPFromRequest(r),
			"url":               r.URL.Path,
			"repository_source": GetTwitchRepositoryFromRequest(r),
			"trace-id":          chitin.GetTraceID(ctx),
			"xray-trace-id":     xray.TraceID(r.Context()),
			"region":            configs.Region(),
		})

		response, err := handler(ctx, values, r)
		if err != nil {
			status := ReportError(ctx, err)
			wErr := ServeError(w, status, err)
			if wErr != nil {
				logx.Error(ctx, err, logx.Fields{ServerError: fmt.Sprintf("Error serving response: %v", wErr)})
			}
			return
		}

		if response == nil {
			w.WriteHeader(http.StatusNoContent)
			return
		}

		wErr := ServeJSON(w, response)
		if wErr != nil {
			logx.Error(ctx, wErr, logx.Fields{ServerError: fmt.Sprintf("Error serving response: %v", wErr)})
		}
	}
}
