package api

import (
	"encoding/json"
	"net/http"
	"net/url"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/internal/image"
	"code.justin.tv/web/users-service/models"
	"goji.io/pat"
	"golang.org/x/net/context"
)

var ErrDuplicateUpdateProperties = &models.CodedError{
	ErrorValue:      "Have duplicate property for the same field to set",
	CodeValue:       "duplicate_update_propertiles",
	StatusCodeValue: http.StatusBadRequest,
}

var ErrMissingDefaultImages = &models.CodedError{
	ErrorValue:      "Missing default images",
	CodeValue:       "missing_default_images",
	StatusCodeValue: http.StatusBadRequest,
}

func (S *Server) setUserImages(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	ID := pat.Param(r, "id")
	if !isValidID(ID) {
		return nil, models.ErrBadIdentifiers
	}

	upToUpdate := &models.ImageProperties{}
	decoder := json.NewDecoder(r.Body)

	defer func() {
		if err := r.Body.Close(); err != nil {
		}
	}()

	err := decoder.Decode(upToUpdate)
	if err != nil {
		return nil, errx.New(errInvalidRequestBody)
	}

	if upToUpdate.ProfileBanner != nil && upToUpdate.DefaultProfileBanner != nil {
		return nil, ErrDuplicateUpdateProperties
	} else if upToUpdate.DefaultProfileBanner != nil {
		if _, ok := image.DefaultProfileBanners[*upToUpdate.DefaultProfileBanner]; !ok {
			return nil, ErrMissingDefaultImages
		}
	}

	if upToUpdate.ProfileImage != nil && upToUpdate.DefaultProfileImage != nil {
		return nil, ErrDuplicateUpdateProperties
	} else if upToUpdate.DefaultProfileImage != nil {
		if _, ok := image.DefaultProfileImages[*upToUpdate.DefaultProfileImage]; !ok {
			return nil, ErrMissingDefaultImages
		}
	}

	if upToUpdate.ChannelOfflineImage != nil && upToUpdate.DefaultChannelOfflineImage != nil {
		return nil, ErrDuplicateUpdateProperties
	} else if upToUpdate.DefaultChannelOfflineImage != nil {
		if _, ok := image.DefaultChannelOfflineImages[*upToUpdate.DefaultChannelOfflineImage]; !ok {
			return nil, ErrMissingDefaultImages
		}
	}

	if upToUpdate.ID == "" {
		upToUpdate.ID = ID
	}

	err = S.logic.SetUserImageProperties(ctx, upToUpdate)
	if err != nil {
		return nil, err
	}

	return nil, nil
}

func (S *Server) uploadUserImage(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	id := pat.Param(r, "id")
	if !isValidID(id) {
		return nil, models.ErrBadIdentifiers
	}

	upToUpdate := &models.UploadableImage{}

	defer func() {
		if err := r.Body.Close(); err != nil {
		}
	}()

	decoder := json.NewDecoder(r.Body)
	err := decoder.Decode(upToUpdate)
	if err != nil {
		return nil, errx.New(errInvalidRequestBody)
	}

	upToUpdate.ID = id
	editorID := upToUpdate.ID
	if upToUpdate.Editor != "" {
		editorID = upToUpdate.Editor
	}

	err = S.auth.AuthorizeUserEdit(ctx, r, editorID)
	if err != nil {
		return nil, err
	}

	uploadInfo, err := S.logic.UploadUserImages(ctx, upToUpdate)

	if values.Get(models.ReturnUploadInfoAsStruct) == "true" {
		return uploadInfo, err
	}

	return uploadInfo.URL, err
}

func (S *Server) setUserImagesAuthed(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	editor := pat.Param(r, "id")
	if !isValidID(editor) {
		return nil, models.ErrBadIdentifiers
	}

	upToUpdate := &models.ImageProperties{}
	decoder := json.NewDecoder(r.Body)

	defer func() {
		if err := r.Body.Close(); err != nil {
		}
	}()

	err := decoder.Decode(upToUpdate)
	if err != nil {
		return nil, errx.New(errInvalidRequestBody)
	}

	if upToUpdate.ProfileBanner != nil && upToUpdate.DefaultProfileBanner != nil {
		return nil, ErrDuplicateUpdateProperties
	} else if upToUpdate.DefaultProfileBanner != nil {
		if _, ok := image.DefaultProfileBanners[*upToUpdate.DefaultProfileBanner]; !ok {
			return nil, ErrMissingDefaultImages
		}
	}

	if upToUpdate.ProfileImage != nil && upToUpdate.DefaultProfileImage != nil {
		return nil, ErrDuplicateUpdateProperties
	} else if upToUpdate.DefaultProfileImage != nil {
		if _, ok := image.DefaultProfileImages[*upToUpdate.DefaultProfileImage]; !ok {
			return nil, ErrMissingDefaultImages
		}
	}

	if upToUpdate.ChannelOfflineImage != nil && upToUpdate.DefaultChannelOfflineImage != nil {
		return nil, ErrDuplicateUpdateProperties
	} else if upToUpdate.DefaultChannelOfflineImage != nil {
		if _, ok := image.DefaultChannelOfflineImages[*upToUpdate.DefaultChannelOfflineImage]; !ok {
			return nil, ErrMissingDefaultImages
		}
	}

	err = S.auth.AuthorizeUserEdit(ctx, r, editor)
	if err != nil {
		return nil, err
	}

	err = S.logic.SetUserImageProperties(ctx, upToUpdate)
	if err != nil {
		return nil, err
	}

	return nil, nil
}
