package api_test

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/web/users-service/api"
	auth "code.justin.tv/web/users-service/internal/auth/mocks"
	"code.justin.tv/web/users-service/internal/image"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/logic/mocks"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type PatchTest struct {
	suite.Suite
	l       *mocks.Logic
	api     *api.Server
	s       *httptest.Server
	decoder *auth.Decoder
}

func (suite *PatchTest) SetupTest() {
	var err error
	suite.l = &mocks.Logic{}
	suite.decoder = &auth.Decoder{}
	suite.api, err = api.NewServer(suite.l, suite.decoder)
	assert.NoError(suite.T(), err)
	suite.s = httptest.NewServer(suite.api)

	err = image.SetUpImageReader()
	assert.NoError(suite.T(), err)
}

func (suite *PatchTest) TestSetImageMetadataSuccess() {
	image := models.ImageProperties{
		ID:           ToSetPropUserID,
		ProfileImage: Users[AdminUserID].ProfileImage,
	}

	suite.l.On("SetUserImageProperties", mock.Anything, &image).Return(nil)
	jsonStr, err := json.Marshal(image)
	assert.NoError(suite.T(), err)
	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%s/images/metadata", suite.s.URL, ToSetPropUserID), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)
	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 204)
}

func (suite *PatchTest) TestUploadImagesNonValidID() {
	up := &models.UploadableImage{
		Type: models.ProfilePictureImageType,
	}

	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%s/images", suite.s.URL, "nonvalid"), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 400)
}

func (suite *PatchTest) TestUploadImagesSuccess() {
	up := &models.UploadableImage{
		ID:   AdminUserID,
		Type: models.ProfilePictureImageType,
	}

	suite.decoder.On("AuthorizeUserEdit", mock.Anything, mock.Anything, AdminUserID).Return(nil)
	suite.l.On("UploadUserImages", mock.Anything, up).Return("validURL", nil)
	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%s/images", suite.s.URL, AdminUserID), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, http.StatusOK)
}

func (suite *PatchTest) TestSetImageMetadataSuccessWithDefaultImage() {
	imageUpdates := models.ImageProperties{
		ID: ToSetPropUserID,
	}

	for key := range image.DefaultProfileImages {
		imageUpdates.DefaultProfileImage = &key
		break
	}

	jsonStr, err := json.Marshal(imageUpdates)
	assert.NoError(suite.T(), err)

	suite.l.On("SetUserImageProperties", mock.Anything, &imageUpdates).Return(nil)
	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%s/images/metadata", suite.s.URL, ToSetPropUserID), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)
	resp, err := http.DefaultClient.Do(req)
	assert.Equal(suite.T(), resp.StatusCode, 204)
}

func (suite *PatchTest) TestSetImageFailWithDuplicateImageInRequest() {
	imageUpdates := models.ImageProperties{
		ID:           ToSetPropUserID,
		ProfileImage: Users[AdminUserID].ProfileImage,
	}

	for key := range image.DefaultProfileImages {
		imageUpdates.DefaultProfileImage = &key
		break
	}

	jsonStr, err := json.Marshal(imageUpdates)
	assert.NoError(suite.T(), err)

	suite.l.On("SetUserImageProperties", mock.Anything, &imageUpdates).Return(nil)
	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%s/images/metadata", suite.s.URL, ToSetPropUserID), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)
	resp, err := http.DefaultClient.Do(req)
	assert.Equal(suite.T(), resp.StatusCode, 400)
}

func (suite *PatchTest) TestSetImageFailWithNonExistingDefaultImage() {
	imageUpdates := models.ImageProperties{
		ID:           ToSetPropUserID,
		ProfileImage: Users[AdminUserID].ProfileImage,
	}

	for key := range image.DefaultProfileImages {
		modifiedKey := key + key
		imageUpdates.DefaultProfileImage = &modifiedKey
		break
	}

	jsonStr, err := json.Marshal(imageUpdates)
	assert.NoError(suite.T(), err)

	suite.l.On("SetUserImageProperties", mock.Anything, &imageUpdates).Return(nil)
	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%s/images/metadata", suite.s.URL, ToSetPropUserID), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)
	resp, err := http.DefaultClient.Do(req)
	assert.Equal(suite.T(), resp.StatusCode, 400)
}

func TestPatchSuite(t *testing.T) {
	suite.Run(t, new(PatchTest))
}
