package api_test

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/web/users-service/api"
	auth "code.justin.tv/web/users-service/internal/auth/mocks"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/logic/mocks"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type PostTest struct {
	suite.Suite
	l       *mocks.Logic
	api     *api.Server
	s       *httptest.Server
	decoder *auth.Decoder
}

func (suite *PostTest) SetupTest() {
	var err error
	suite.l = &mocks.Logic{}
	suite.decoder = &auth.Decoder{}
	suite.api, err = api.NewServer(suite.l, suite.decoder)
	assert.NoError(suite.T(), err)
	suite.s = httptest.NewServer(suite.api)
}

func (suite *PostTest) TestCreateUserSuccess() {
	up := &models.CreateUserProperties{
		Login:    *Users[AdminUserID].Login,
		Birthday: GetBirthday(AdminUserID),
		Email:    *Users[AdminUserID].Email,
	}
	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	suite.l.On("CreateUser", mock.Anything, mock.Anything).Return(Users[AdminUserID], nil)

	req, err := http.NewRequest("POST", fmt.Sprintf("%v/users", suite.s.URL), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 200)

	suite.l.AssertExpectations(suite.T())
}

func (suite *PostTest) TestCreateUserMissingLogin() {
	up := &models.CreateUserProperties{
		Birthday: GetBirthday(AdminUserID),
	}
	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	req, err := http.NewRequest("POST", fmt.Sprintf("%v/users", suite.s.URL), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 400)
}

func (suite *PostTest) TestCreateUserMissingEmail() {
	up := &models.CreateUserProperties{
		Login:    *Users[AdminUserID].Login,
		Birthday: GetBirthday(AdminUserID),
	}
	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	req, err := http.NewRequest("POST", fmt.Sprintf("%v/users", suite.s.URL), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 400)
}

func TestPostSuite(t *testing.T) {
	suite.Run(t, new(PostTest))
}
