package api

import (
	"encoding/json"
	"net/http"
	"net/url"
	"strings"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/web/users-service/models"
	"goji.io/pat"
	"golang.org/x/net/context"
	"gopkg.in/validator.v2"
)

var errInvalidRequestBody = &models.CodedError{
	ErrorValue:      "Request body doesn't contain valid JSON",
	CodeValue:       "invalid_body",
	StatusCodeValue: http.StatusBadRequest,
}

// banUser will ban a given user as specified by the ID parameter in the route.
func (S *Server) banUser(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	banProp := &models.BanUserProperties{}
	decoder := json.NewDecoder(r.Body)
	err := decoder.Decode(banProp)
	if err != nil {
		return nil, errx.New(errInvalidRequestBody)
	}

	IDParam := pat.Param(r, "id")
	if !isValidID(IDParam) {
		return nil, models.ErrBadIdentifiers
	}

	banProp.Type = strings.ToLower(banProp.Type)
	if banProp.TargetUserID == "" || banProp.TargetUserID == "0" {
		banProp.TargetUserID = IDParam
	}
	if err := validator.Validate(banProp); err != nil {
		return nil, ErrNoParameters
	}

	err = S.logic.BanUser(ctx, banProp)
	// Temp debugging line for failing ban user issues
	if err != nil {
		jsonBody, marshallingErr := json.Marshal(banProp)
		if marshallingErr != nil {
			jsonBody = []byte{}
		}
		logx.Error(ctx, "failed to ban user", logx.Fields{
			"json_body": string(jsonBody),
			"error": err.Error(),
		})
	}

	return nil, err
}

func (S *Server) setProperties(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	ID := pat.Param(r, "id")
	if !isValidID(ID) {
		return nil, models.ErrBadIdentifiers
	}

	upToUpdate := &models.UpdateableProperties{}

	decoder := json.NewDecoder(r.Body)
	err := decoder.Decode(upToUpdate)
	if err != nil {
		return nil, errx.New(errInvalidRequestBody)
	}

	if upToUpdate.Email != nil {
		*upToUpdate.Email = strings.ToLower(*upToUpdate.Email)
	}

	err = S.logic.SetUserProperties(ctx, ID, upToUpdate)
	if err != nil {
		return nil, err
	}

	return nil, nil
}

func (S *Server) addDMCAStrike(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	ID := pat.Param(r, "id")
	if !isValidID(ID) {
		return nil, models.ErrBadIdentifiers
	}

	err := S.logic.AlterDMCAStrike(ctx, ID, 1)
	if err != nil {
		return nil, err
	}

	return nil, nil
}

func (S *Server) removeDMCAStrike(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	ID := pat.Param(r, "id")
	if !isValidID(ID) {
		return nil, models.ErrBadIdentifiers
	}

	err := S.logic.AlterDMCAStrike(ctx, ID, -1)
	if err != nil {
		return nil, err
	}

	return nil, nil
}
