package api_test

import (
	"bytes"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	. "github.com/stretchr/testify/mock"

	"code.justin.tv/web/users-service/api"
	"code.justin.tv/web/users-service/backend/util"
	auth "code.justin.tv/web/users-service/internal/auth/mocks"
	"code.justin.tv/web/users-service/logic/mocks"
	"code.justin.tv/web/users-service/models"
)

func TestLoginRename(t *testing.T) {
	Convey("Requesting login change", t, func() {
		l := &mocks.Logic{}
		decoder := &auth.Decoder{}

		api, err := api.NewServer(l, decoder)
		So(err, ShouldBeNil)

		s := httptest.NewServer(api)
		defer s.Close()

		id := "45573555"
		newLogin := "egflores"
		Convey("with only submiting new_login field", func() {
			up := &models.UpdateableProperties{
				NewLogin: &newLogin,
			}
			var jsonStr = []byte(`{"new_login":"egflores"}`)

			l.On("SetUserProperties", Anything, id, up).Return(nil)

			req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%s", s.URL, id), bytes.NewBuffer(jsonStr))
			So(err, ShouldBeNil)

			resp, err := http.DefaultClient.Do(req)
			So(err, ShouldBeNil)

			l.AssertExpectations(t)

			Convey("the response has the correct HTTP status code", func() {
				So(resp.Status, ShouldEqual, "204 No Content")
			})
		})
		Convey("to a login that already exists", func() {
			up := &models.UpdateableProperties{
				NewLogin: &newLogin,
			}
			var jsonStr = []byte(`{"new_login":"egflores"}`)

			l.On("SetUserProperties", Anything, id, up).Return(models.ErrLoginNotAvailable)

			req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%s", s.URL, id), bytes.NewBuffer(jsonStr))
			So(err, ShouldBeNil)

			resp, err := http.DefaultClient.Do(req)
			So(err, ShouldBeNil)

			l.AssertExpectations(t)

			Convey("the response has the correct HTTP status code", func() {
				So(resp.Status, ShouldEqual, "403 Forbidden")
			})
		})
		Convey("to a login that already exists as a displayname", func() {
			up := &models.UpdateableProperties{
				NewLogin: &newLogin,
			}
			var jsonStr = []byte(`{"new_login":"egflores"}`)

			l.On("SetUserProperties", Anything, id, up).Return(models.ErrDisplaynameNotAvailable)

			req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%s", s.URL, id), bytes.NewBuffer(jsonStr))
			So(err, ShouldBeNil)

			resp, err := http.DefaultClient.Do(req)
			So(err, ShouldBeNil)

			l.AssertExpectations(t)

			Convey("the response has the correct HTTP status code", func() {
				So(resp.Status, ShouldEqual, "403 Forbidden")
			})
		})
		Convey("while still experiencing cooldown", func() {
			skipLogin := false
			up := &models.UpdateableProperties{
				NewLogin:          &newLogin,
				SkipLoginCooldown: &skipLogin,
			}
			var jsonStr = []byte(`{"new_login":"egflores", "skip_login_cooldown":false}`)

			l.On("SetUserProperties", Anything, id, up).Return(models.ErrNotAllowedToChangeLogin)

			req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%s", s.URL, id), bytes.NewBuffer(jsonStr))
			So(err, ShouldBeNil)

			resp, err := http.DefaultClient.Do(req)
			So(err, ShouldBeNil)

			l.AssertExpectations(t)

			Convey("the response has the correct HTTP status code", func() {
				So(resp.Status, ShouldEqual, "403 Forbidden")
			})
		})
		Convey("with a user with no properties", func() {
			up := &models.UpdateableProperties{
				NewLogin: &newLogin,
			}
			var jsonStr = []byte(`{"new_login":"egflores"}`)

			l.On("SetUserProperties", Anything, id, up).Return(util.ErrNoProperties)

			req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/users/%s", s.URL, id), bytes.NewBuffer(jsonStr))
			So(err, ShouldBeNil)

			resp, err := http.DefaultClient.Do(req)
			So(err, ShouldBeNil)

			l.AssertExpectations(t)

			Convey("the response has the correct HTTP status code", func() {
				So(resp.Status, ShouldEqual, "404 Not Found")
			})
		})
	})
}
