package api

import (
	"encoding/json"
	"net/http"
	"net/url"

	"time"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/models"
	"goji.io/pat"
	"golang.org/x/net/context"
	"gopkg.in/validator.v2"
)

func (S *Server) getReservations(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	logins := values["login"]
	validLogins := []string{}
	if len(logins) == 0 {
		return nil, ErrBadParameters
	}

	for _, login := range logins {
		if b := isValidLogin(login); b {
			validLogins = append(validLogins, login)
		}
	}

	return S.logic.GetReservations(ctx, logins)
}

func (S *Server) addReservation(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	prop := &models.ReservationProperties{}
	decoder := json.NewDecoder(r.Body)
	err := decoder.Decode(prop)
	if err != nil {
		return nil, errx.New(errInvalidRequestBody)
	}

	if err := validator.Validate(prop); err != nil {
		return nil, ErrBadParameters
	}

	if prop.ExpiresOn != nil && (*prop.ExpiresOn).Before(time.Now()) {
		return nil, ErrBadParameters
	}

	if b := isValidLogin(prop.Login); !b {
		return nil, ErrBadParameters
	}

	return nil, S.logic.AddReservation(ctx, *prop)
}

func (S *Server) updateReservation(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	prop := &models.ReservationProperties{}
	decoder := json.NewDecoder(r.Body)
	err := decoder.Decode(prop)
	if err != nil {
		return nil, errx.New(errInvalidRequestBody)
	}

	if err := validator.Validate(prop); err != nil {
		return nil, ErrBadParameters
	}

	if prop.ExpiresOn != nil && (*prop.ExpiresOn).Before(time.Now()) {
		return nil, ErrBadParameters
	}

	if b := isValidLogin(prop.Login); !b {
		return nil, ErrBadParameters
	}

	return nil, S.logic.UpdateReservation(ctx, *prop)
}

func (S *Server) deleteReservation(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	login := pat.Param(r, "login")
	if b := isValidLogin(login); !b {
		return nil, ErrBadParameters
	}

	return nil, S.logic.DeleteReservation(ctx, login)
}
