package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"errors"

	logicMocks "code.justin.tv/web/users-service/logic/mocks"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestGetReservation(t *testing.T) {
	propOne := models.ReservationProperties{}
	propTwo := models.ReservationProperties{}
	result := models.ReservationPropertiesResult{Results: []models.ReservationProperties{propOne, propTwo}}
	for _, scenario := range []struct {
		Name   string
		Setup  func(t *testing.T, api *Server, logic *logicMocks.Logic)
		Status int
		URL    string
	}{
		{
			Name: "happy path",
			Setup: func(t *testing.T, api *Server, logic *logicMocks.Logic) {
				logic.On("GetReservations", mock.Anything, []string{"user123", "user456"}).Return(result, nil)
			},
			Status: 200,
			URL:    "login=user123&login=user456",
		},
		{
			Name: "happy path",
			Setup: func(t *testing.T, api *Server, logic *logicMocks.Logic) {
				logic.On("GetReservations", mock.Anything, []string{"user123", "user456"}).Return(result, errors.New("unexpected error"))
			},
			Status: 500,
			URL:    "login=user123&login=user456",
		},
		{
			Name:   "missing logins in request",
			Setup:  func(t *testing.T, api *Server, logic *logicMocks.Logic) {},
			Status: 400,
			URL:    "",
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			logicMock := &logicMocks.Logic{}
			api, err := NewServer(logicMock, nil)

			assert.NoError(t, err)
			server := httptest.NewServer(api)

			if scenario.Setup != nil {
				scenario.Setup(t, api, logicMock)
			}

			req, err := http.NewRequest("GET", fmt.Sprintf("%v/reservations?%v", server.URL, scenario.URL), nil)
			assert.NoError(t, err)

			resp, err := http.DefaultClient.Do(req)
			assert.NoError(t, err)
			assert.Equal(t, resp.StatusCode, scenario.Status)

			logicMock.AssertExpectations(t)
		})
	}
}

func TestUpdateReservation(t *testing.T) {
	login := "reserved_login"
	reservationType := "JTV Reservation"
	pastTime := time.Now().Truncate(time.Hour)
	err := errors.New("unexpected error")
	prop := models.ReservationProperties{
		Login:     login,
		Type:      reservationType,
		Reason:    "Test",
		ExpiresOn: nil,
	}
	for _, scenario := range []struct {
		Name   string
		Setup  func(t *testing.T, api *Server, logic *logicMocks.Logic)
		Status int
		Prop   models.ReservationProperties
	}{
		{
			Name: "happy path",
			Setup: func(t *testing.T, api *Server, logic *logicMocks.Logic) {
				logic.On("UpdateReservation", mock.Anything, prop).Return(nil)
			},
			Status: 204,
			Prop:   prop,
		},
		{
			Name: "backend is returning unexpected error",
			Setup: func(t *testing.T, api *Server, logic *logicMocks.Logic) {
				logic.On("UpdateReservation", mock.Anything, prop).Return(err)
			},
			Status: 500,
			Prop:   prop,
		},
		{
			Name:   "has bad type",
			Setup:  func(t *testing.T, api *Server, logic *logicMocks.Logic) {},
			Status: 400,
			Prop: models.ReservationProperties{
				Login:     login,
				Type:      "Bad Type",
				Reason:    "Test",
				ExpiresOn: nil,
			},
		},
		{
			Name:   "has empty login",
			Setup:  func(t *testing.T, api *Server, logic *logicMocks.Logic) {},
			Status: 400,
			Prop: models.ReservationProperties{
				Login:     "",
				Type:      reservationType,
				Reason:    "Test",
				ExpiresOn: nil,
			},
		},
		{
			Name: "have the other valid reservation type",
			Setup: func(t *testing.T, api *Server, logic *logicMocks.Logic) {
				logic.On("UpdateReservation", mock.Anything, models.ReservationProperties{
					Login:     login,
					Type:      "Reserved Record",
					Reason:    "Test",
					ExpiresOn: nil,
				}).Return(nil)
			},
			Status: 204,
			Prop: models.ReservationProperties{
				Login:     login,
				Type:      "Reserved Record",
				Reason:    "Test",
				ExpiresOn: nil,
			},
		},
		{
			Name:   "have a past expire time",
			Setup:  func(t *testing.T, api *Server, logic *logicMocks.Logic) {},
			Status: 400,
			Prop: models.ReservationProperties{
				Login:     "",
				Type:      reservationType,
				Reason:    "Test",
				ExpiresOn: &pastTime,
			},
		},
		{
			Name:   "reason empty",
			Setup:  func(t *testing.T, api *Server, logic *logicMocks.Logic) {},
			Status: 400,
			Prop: models.ReservationProperties{
				Login:     login,
				Type:      "Bad Type",
				Reason:    "",
				ExpiresOn: nil,
			},
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			logicMock := &logicMocks.Logic{}
			api, err := NewServer(logicMock, nil)

			assert.NoError(t, err)
			server := httptest.NewServer(api)

			if scenario.Setup != nil {
				scenario.Setup(t, api, logicMock)
			}

			jsonStr, err := json.Marshal(scenario.Prop)
			assert.NoError(t, err)

			req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/reservations", server.URL), bytes.NewBuffer(jsonStr))
			assert.NoError(t, err)

			resp, err := http.DefaultClient.Do(req)
			assert.NoError(t, err)
			assert.Equal(t, resp.StatusCode, scenario.Status)

			logicMock.AssertExpectations(t)
		})
	}
}

func TestAddReservation(t *testing.T) {
	login := "reserved_login"
	reservationType := "JTV Reservation"
	pastTime := time.Now().Truncate(time.Hour)
	err := errors.New("unexpected error")
	prop := models.ReservationProperties{
		Login:     login,
		Type:      reservationType,
		Reason:    "Test",
		ExpiresOn: nil,
	}
	for _, scenario := range []struct {
		Name   string
		Setup  func(t *testing.T, api *Server, logic *logicMocks.Logic)
		Status int
		Prop   models.ReservationProperties
	}{
		{
			Name: "happy path",
			Setup: func(t *testing.T, api *Server, logic *logicMocks.Logic) {
				logic.On("AddReservation", mock.Anything, prop).Return(nil)
			},
			Status: 204,
			Prop:   prop,
		},
		{
			Name: "backend is returning unexpected error",
			Setup: func(t *testing.T, api *Server, logic *logicMocks.Logic) {
				logic.On("AddReservation", mock.Anything, prop).Return(err)
			},
			Status: 500,
			Prop:   prop,
		},
		{
			Name:   "has bad type",
			Setup:  func(t *testing.T, api *Server, logic *logicMocks.Logic) {},
			Status: 400,
			Prop: models.ReservationProperties{
				Login:     login,
				Type:      "Bad Type",
				Reason:    "Test",
				ExpiresOn: nil,
			},
		},
		{
			Name:   "has empty login",
			Setup:  func(t *testing.T, api *Server, logic *logicMocks.Logic) {},
			Status: 400,
			Prop: models.ReservationProperties{
				Login:     "",
				Type:      reservationType,
				Reason:    "Test",
				ExpiresOn: nil,
			},
		},
		{
			Name: "have the other valid reservation type",
			Setup: func(t *testing.T, api *Server, logic *logicMocks.Logic) {
				logic.On("AddReservation", mock.Anything, models.ReservationProperties{
					Login:     login,
					Type:      "Reserved Record",
					Reason:    "Test",
					ExpiresOn: nil,
				}).Return(nil)
			},
			Status: 204,
			Prop: models.ReservationProperties{
				Login:     login,
				Type:      "Reserved Record",
				Reason:    "Test",
				ExpiresOn: nil,
			},
		},
		{
			Name:   "have a past expire time",
			Setup:  func(t *testing.T, api *Server, logic *logicMocks.Logic) {},
			Status: 400,
			Prop: models.ReservationProperties{
				Login:     "",
				Type:      reservationType,
				Reason:    "Test",
				ExpiresOn: &pastTime,
			},
		},
		{
			Name:   "reason empty",
			Setup:  func(t *testing.T, api *Server, logic *logicMocks.Logic) {},
			Status: 400,
			Prop: models.ReservationProperties{
				Login:     login,
				Type:      "Bad Type",
				Reason:    "",
				ExpiresOn: nil,
			},
		},
	} {
		t.Run(scenario.Name, func(t *testing.T) {
			logicMock := &logicMocks.Logic{}
			api, err := NewServer(logicMock, nil)

			assert.NoError(t, err)
			server := httptest.NewServer(api)

			if scenario.Setup != nil {
				scenario.Setup(t, api, logicMock)
			}

			jsonStr, err := json.Marshal(scenario.Prop)
			assert.NoError(t, err)

			req, err := http.NewRequest("POST", fmt.Sprintf("%v/reservations", server.URL), bytes.NewBuffer(jsonStr))
			assert.NoError(t, err)

			resp, err := http.DefaultClient.Do(req)
			assert.NoError(t, err)
			assert.Equal(t, resp.StatusCode, scenario.Status)

			logicMock.AssertExpectations(t)
		})
	}
}
