package api

import (
	"encoding/json"
	"net/http"
	"net/url"
	"strconv"

	"strings"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/models"
	"code.justin.tv/web/users-service/validators"
	"goji.io/pat"
	"golang.org/x/net/context"
	"gopkg.in/validator.v2"
)

func (S *Server) setChannelProperties(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	up := &models.UpdateChannelProperties{}
	decoder := json.NewDecoder(r.Body)
	err := decoder.Decode(up)
	if err != nil {
		return nil, errx.New(errInvalidRequestBody)
	}

	if !isValidSizedID(up.ID) {
		return nil, models.ErrBadChannelIdentifiers
	}

	if err := validator.Validate(up); err != nil {
		return nil, ErrNoParameters
	}

	if up.RedirectChannel != nil && !validators.NameRegex.MatchString(strings.ToLower(*up.RedirectChannel)) && *up.RedirectChannel != "" {
		return nil, models.ErrBadChannelIdentifiers
	}

	return nil, S.logic.SetChannelProperties(ctx, up)
}

func (S *Server) setChannelPropertiesExternal(ctx context.Context, values url.Values, r *http.Request) (interface{}, error) {
	id := pat.Param(r, "id")

	if !isValidID(id) {
		return nil, models.ErrBadIdentifiers
	}

	up := &models.UpdateChannelProperties{}
	decoder := json.NewDecoder(r.Body)
	err := decoder.Decode(up)
	if err != nil {
		return nil, errx.New(errInvalidRequestBody)
	}

	if !isValidSizedID(up.ID) {
		return nil, models.ErrBadChannelIdentifiers
	}

	err = S.auth.AuthorizeChannelEdit(ctx, r, id, strconv.FormatUint(up.ID, 10))
	if err != nil {
		return nil, err
	}

	if err := validator.Validate(up); err != nil {
		return nil, ErrNoParameters
	}

	return nil, S.logic.SetBasicChannelProperties(ctx, up)
}
