package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/web/users-service/internal/auth"
	authmocks "code.justin.tv/web/users-service/internal/auth/mocks"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/logic/mocks"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
)

type SetChannelsTest struct {
	suite.Suite
	l       *mocks.Logic
	api     *Server
	s       *httptest.Server
	decoder *authmocks.Decoder
}

func (suite *SetChannelsTest) SetupTest() {
	var err error
	suite.l = &mocks.Logic{}
	suite.decoder = &authmocks.Decoder{}
	suite.api, err = NewServer(suite.l, suite.decoder)
	assert.NoError(suite.T(), err)
	suite.s = httptest.NewServer(suite.api)
}

func (suite *SetChannelsTest) TestSetChannelSuccess() {
	redirectChannel := "testChannel"
	up := &models.UpdateChannelProperties{
		ID:              Channels[AdminUserID].ID,
		Status:          Channels[ToSetPropUserID].Status,
		RedirectChannel: &redirectChannel,
	}
	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	suite.l.On("SetChannelProperties", mock.Anything, mock.Anything).Return(nil, nil)

	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/channels", suite.s.URL), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 204)

	suite.l.AssertExpectations(suite.T())
}

func (suite *SetChannelsTest) TestSetChannelExternalSuccess() {
	up := &models.UpdateChannelProperties{
		ID:     Channels[AdminUserID].ID,
		Status: Channels[ToSetPropUserID].Status,
	}
	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	suite.l.On("SetBasicChannelProperties", mock.Anything, mock.Anything).Return(nil, nil)
	suite.decoder.On("AuthorizeChannelEdit", mock.Anything, mock.Anything, AdminUserID, AdminUserID).Return(nil)

	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/channels/editor/51756334", suite.s.URL), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 204)

	suite.l.AssertExpectations(suite.T())
}

func (suite *SetChannelsTest) TestSetChannelWithInvalidRedirect() {
	badRedirect := "p/help"
	up := &models.UpdateChannelProperties{
		ID:              Channels[AdminUserID].ID,
		RedirectChannel: &badRedirect,
	}
	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/channels", suite.s.URL), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 400)
}

func (suite *SetChannelsTest) TestSetChannelsMissingID() {
	up := &models.UpdateChannelProperties{
		Status: Channels[ToSetPropUserID].Status,
	}
	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/channels", suite.s.URL), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 400)
}

func (suite *SetChannelsTest) TestSetChannelsIDTooBig() {
	up := &models.UpdateChannelProperties{
		ID:     maxID + 2,
		Status: Channels[ToSetPropUserID].Status,
	}
	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/channels", suite.s.URL), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 400)
}

func (suite *SetChannelsTest) TestSetChannelExternalBadEditorID() {
	up := &models.UpdateChannelProperties{
		ID:     Channels[AdminUserID].ID,
		Status: Channels[ToSetPropUserID].Status,
	}
	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	suite.l.On("SetBasicChannelProperties", mock.Anything, mock.Anything).Return(nil, nil)
	suite.decoder.On("AuthorizeChannelEdit", mock.Anything, mock.Anything, AdminUserID, Channels[AdminUserID].ID).Return(nil)

	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/channels/editor/posd0fss", suite.s.URL), bytes.NewBuffer(jsonStr))

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 400)
}

func (suite *SetChannelsTest) TestSetChannelExternalIDTooBig() {
	up := &models.UpdateChannelProperties{
		ID:     maxID + 2,
		Status: Channels[ToSetPropUserID].Status,
	}
	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	suite.l.On("SetBasicChannelProperties", mock.Anything, mock.Anything).Return(nil, nil)
	suite.decoder.On("AuthorizeChannelEdit", mock.Anything, mock.Anything, AdminUserID, Channels[AdminUserID].ID).Return(nil)

	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/channels/editor/1", suite.s.URL), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, 400)
}

func (suite *SetChannelsTest) TestSetChannelExternalForbidden() {
	up := &models.UpdateChannelProperties{
		ID:     Channels[AdminUserID].ID,
		Status: Channels[ToSetPropUserID].Status,
	}
	jsonStr, err := json.Marshal(up)
	assert.NoError(suite.T(), err)

	suite.l.On("SetBasicChannelProperties", mock.Anything, mock.Anything).Return(nil, nil)
	suite.decoder.On("AuthorizeChannelEdit", mock.Anything, mock.Anything, AdminUserID, AdminUserID).Return(auth.AuthFailureError)

	req, err := http.NewRequest("PATCH", fmt.Sprintf("%v/channels/editor/51756334", suite.s.URL), bytes.NewBuffer(jsonStr))
	assert.NoError(suite.T(), err)

	resp, err := http.DefaultClient.Do(req)
	assert.NoError(suite.T(), err)
	assert.Equal(suite.T(), resp.StatusCode, http.StatusForbidden)
}

func TestSetChannelsTestSuite(t *testing.T) {
	suite.Run(t, new(SetChannelsTest))
}
