package api

import (
	"encoding/json"
	"fmt"
	"net"
	"net/http"
	"net/url"
	"regexp"
	"strconv"
	"strings"

	"code.justin.tv/chat/db"
	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/foundation/gomemcache/memcache"
	"code.justin.tv/web/users-service/models"
	"github.com/afex/hystrix-go/hystrix"
	"github.com/garyburd/redigo/redis"
	"golang.org/x/net/context"
)

const (
	ServerError   = "server_err"
	InternalError = "internal_error"
)

var (
	_ ServiceError = (*models.CodedError)(nil)
)

// MaxID is the largest value a PostgreSQL 'integer' type can have.
// https://www.postgresql.org/docs/9.1/static/datatype-numeric.html
const maxID = 2147483647

func isValidID(s string) bool {
	id, err := strconv.ParseUint(s, 10, 64)
	return err == nil && isValidSizedID(id)
}

func isValidSizedID(id uint64) bool {
	return id <= maxID
}

func isValidLogin(s string) bool {
	m, err := regexp.MatchString("^[a-zA-Z0-9][a-zA-Z0-9_-]*$", s)
	return m && err == nil
}

func RequireParams(form url.Values, params []string) error {
	for _, param := range params {
		if len(form[param]) == 0 {
			return fmt.Errorf("Missing param: %s", param)
		}
	}
	return nil
}

func RequireFormParams(r *http.Request, params []string) error {
	for _, param := range params {
		if len(r.FormValue(param)) == 0 {
			return fmt.Errorf("Missing param: %s", param)
		}
	}
	return nil
}

func ServeJSON(w http.ResponseWriter, v interface{}) error {
	content, err := json.Marshal(v)
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return err
	}
	w.Header().Set("Content-Length", strconv.Itoa(len(content)))
	w.Header().Set("Content-Type", "application/json;charset=utf-8")
	_, err = w.Write(content)
	return err
}

func ServeError(w http.ResponseWriter, status int, err error) error {
	errRes := NewErrorResponse(status, err)
	w.WriteHeader(errRes.Status)
	return ServeJSON(w, errRes)
}

func ReportError(ctx context.Context, err error) int {
	logToRollbar := false
	status := http.StatusInternalServerError

	if e, ok := errx.Unwrap(err).(ServiceError); ok {
		logToRollbar = e.ShouldLog()
		status = e.StatusCode()
	} else {
		logToRollbar = true
		switch errx.Unwrap(err) {
		case db.ErrConnPoolExhausted, redis.ErrPoolExhausted, memcache.ErrServerError:
			status = http.StatusServiceUnavailable
		case hystrix.ErrCircuitOpen, hystrix.ErrTimeout, hystrix.ErrMaxConcurrency:
			logToRollbar = false
		}
	}

	if logToRollbar {
		logx.Error(ctx, err)
	}
	return status
}

func GetTwitchRepositoryFromRequest(r *http.Request) string {
	source := r.Header.Get("Twitch-Repository")
	if source == "" {
		source = "unknown"
	}
	return source
}

func GetIPFromRequest(r *http.Request) string {
	ipString := fetchLastHeader(r, "X-Forwarded-For")
	if ipString == "" {
		host, _, err := net.SplitHostPort(r.RemoteAddr)
		if err != nil {
			return ""
		} else {
			ipString = host
		}
	}

	ip := net.ParseIP(ipString)
	if ip == nil {
		return ""
	}

	return ipString
}

func fetchHeaderValues(r *http.Request, name string) []string {
	h := r.Header.Get(name)
	if h == "" {
		return nil
	}
	return strings.Split(h, ",")
}

func fetchLastHeader(r *http.Request, name string) string {
	values := fetchHeaderValues(r, name)
	if len(values) == 0 {
		return ""
	}

	return strings.TrimSpace(values[len(values)-1])
}

func NewErrorResponse(status int, err error) models.ErrorResponse {
	statusText := http.StatusText(status)
	if statusText == "" {
		statusText = ExtentionStatusText(status)
	}
	return models.ErrorResponse{
		Status:     status,
		Message:    err.Error(),
		StatusText: statusText,
		ErrorCode:  ErrorCode(err),
	}
}

func ExtentionStatusText(code int) string {
	return extentionStatusText[code]
}

// extentionStatusText supports extra status codes that the stdlib http package does not.
var extentionStatusText = map[int]string{
	http.StatusUnprocessableEntity: "Unprocessable entity",
}

func ErrorCode(err error) string {
	if e, ok := errx.Unwrap(err).(ServiceError); ok {
		return e.Code()
	} else {
		return InternalError
	}
}

// ServiceError is used server side to describe an error. It defines it's code, whether it needs to be logged,
// an error message, and it's status code. All errors being returned by the API should implement this.
type ServiceError interface {
	// Short-hand code for the error
	Code() string
	// Whether or not
	ShouldLog() bool

	// Methods below are expected client side by ClientError
	// Full error message
	Error() string
	// HTTP status code
	StatusCode() int
}
