package channels

import (
	"fmt"
	"net/http"
	"strings"

	"golang.org/x/net/context"

	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/database"
	"code.justin.tv/web/users-service/models"
)

// ErrNoChannelProperties indicates that there are no properties for the channel name given
var ErrNoChannelProperties = &models.CodedError{
	ErrorValue:      "No properties found for this channel name",
	CodeValue:       "no_channel_properties",
	StatusCodeValue: http.StatusNotFound,
}

type ChannelPropertiesResult struct {
	Results []models.ChannelProperties `json:"results"`
}

type ChannelQuery struct {
	sql    string
	params []interface{}
}

//go:generate mockery -name Backend
type Backend interface {
	GetAllChannelPropertiesBulk(ctx context.Context, channelIDs []uint64, channelNames []string, opts util.ReadOptions) ([]models.ChannelProperties, error)
	UpdateChannelProperties(ctx context.Context, updateProps models.UpdateChannelProperties) error
	UpdateBasicChannelProperties(ctx context.Context, updateProps models.UpdateChannelProperties) error
	DeleteChannel(ctx context.Context, channelID string) error
	GetChannelsByRedirectChannel(ctx context.Context, name string) ([]models.ChannelProperties, error)
	ExpireChannelProperties(ctx context.Context, prop *models.ChannelProperties) error
}

type channelPropertiesRepoImpl struct {
	sdb database.Querier
	mdb database.Querier
}

func BuildQuery(channelIDs []uint64, channelNames []string) (ChannelQuery, error) {

	var querySuffix []string
	var queryParams []interface{}

	if len(channelIDs) > 0 {

		var idSuffix []string

		for i, id := range channelIDs {
			idSuffix = append(idSuffix, fmt.Sprintf("$%d", i+1))
			queryParams = append(queryParams, id)
		}
		querySuffix = append(querySuffix, fmt.Sprintf("u.id IN (%s)", strings.Join(idSuffix, ",")))
	}

	if len(channelNames) > 0 {

		var nameSuffix []string

		for i, name := range channelNames {
			nameSuffix = append(nameSuffix, fmt.Sprintf("$%d", i+1+len(channelIDs)))
			queryParams = append(queryParams, name)
		}
		querySuffix = append(querySuffix, fmt.Sprintf("login IN (%s)", strings.Join(nameSuffix, ",")))
	}

	query := strings.Join([]string{sqlSelectPropertiesBulkPartial, strings.Join(querySuffix, " OR ")}, " ")

	return ChannelQuery{sql: query, params: queryParams}, nil
}

// Placeholder to satisfy shared interface
func (c *channelPropertiesRepoImpl) ExpireChannelProperties(ctx context.Context, prop *models.ChannelProperties) error {
	return nil
}

func NewBackend(sdb database.Querier, mdb database.Querier) (Backend, error) {
	return &channelPropertiesRepoImpl{
		sdb: sdb,
		mdb: mdb,
	}, nil
}
