package channels

import (
	"database/sql"
	"errors"
	"testing"

	"golang.org/x/net/context"

	. "github.com/smartystreets/goconvey/convey"
	. "github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/database/mocks"
)

func TestStart(t *testing.T) {
	Convey("When calling GetAllChannelPropertiesBulk", t, func() {
		db := &mocks.Querier{}
		mdb := &mocks.Querier{}
		cpr := &channelPropertiesRepoImpl{db, mdb}
		ctx := context.Background()

		Convey("with only a login name", func() {
			login := []string{"dadbot"}

			Convey("and the DB returns a valid row", func() {
				rows := &mocks.Rows{}

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(nil)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				channelQuery, err := BuildQuery(nil, login)
				So(err, ShouldBeNil)

				db.On("Query", ctx, "channel_properties_bulk", channelQuery.sql, channelQuery.params).Return(rows, nil)

				properties, err := cpr.GetAllChannelPropertiesBulk(ctx, nil, login, util.ReadOptions{})
				So(properties, ShouldNotBeNil)
				So(err, ShouldBeNil)
			})

			Convey("and the DB returns no row", func() {
				rows := &mocks.Rows{}

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(sql.ErrNoRows)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				channelQuery, err := BuildQuery(nil, login)
				So(err, ShouldBeNil)

				db.On("Query", ctx, "channel_properties_bulk", channelQuery.sql, channelQuery.params).Return(rows, nil)

				properties, err := cpr.GetAllChannelPropertiesBulk(ctx, nil, login, util.ReadOptions{})
				So(properties, ShouldBeNil)
			})

			Convey("and the DB returns an error", func() {
				rows := &mocks.Rows{}
				dbErr := errors.New("this should never happen")

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(dbErr)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				channelQuery, err := BuildQuery(nil, login)
				So(err, ShouldBeNil)

				db.On("Query", ctx, "channel_properties_bulk", channelQuery.sql, channelQuery.params).Return(rows, nil)

				properties, err := cpr.GetAllChannelPropertiesBulk(ctx, nil, login, util.ReadOptions{})
				So(properties, ShouldBeNil)
				So(errx.Unwrap(err), ShouldEqual, dbErr)
			})
		})

		Convey("with only an id", func() {
			id := []uint64{5}

			Convey("and the DB returns a valid row", func() {
				rows := &mocks.Rows{}
				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(nil)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				channelQuery, err := BuildQuery(id, nil)
				So(err, ShouldBeNil)

				db.On("Query", ctx, "channel_properties_bulk", channelQuery.sql, channelQuery.params).Return(rows, nil)

				properties, err := cpr.GetAllChannelPropertiesBulk(ctx, id, nil, util.ReadOptions{})
				So(properties, ShouldNotBeNil)
				So(err, ShouldBeNil)
			})

			Convey("and the DB returns no row", func() {
				rows := &mocks.Rows{}

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(sql.ErrNoRows)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				channelQuery, err := BuildQuery(id, nil)
				So(err, ShouldBeNil)

				db.On("Query", ctx, "channel_properties_bulk", channelQuery.sql, channelQuery.params).Return(rows, nil)

				properties, err := cpr.GetAllChannelPropertiesBulk(ctx, id, nil, util.ReadOptions{})
				So(properties, ShouldBeNil)
			})

			Convey("and the DB returns an error", func() {
				rows := &mocks.Rows{}
				dbErr := errors.New("this should never happen")

				rows.On("Next", Anything).Return(true).Once()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(dbErr)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				channelQuery, err := BuildQuery(id, nil)
				So(err, ShouldBeNil)

				db.On("Query", ctx, "channel_properties_bulk", channelQuery.sql, channelQuery.params).Return(rows, nil)

				properties, err := cpr.GetAllChannelPropertiesBulk(ctx, id, nil, util.ReadOptions{})
				So(properties, ShouldBeNil)
				So(errx.Unwrap(err), ShouldEqual, dbErr)
			})
		})

		Convey("with multiple login names", func() {
			logins := []string{"dadbot", "creative"}

			Convey("and the DB returns valid rows", func() {
				rows := &mocks.Rows{}
				rows.On("Next", Anything).Return(true).Twice()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(nil)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				channelQuery, err := BuildQuery(nil, logins)
				So(err, ShouldBeNil)

				db.On("Query", ctx, "channel_properties_bulk", channelQuery.sql, channelQuery.params).Return(rows, nil)

				properties, err := cpr.GetAllChannelPropertiesBulk(ctx, nil, logins, util.ReadOptions{})
				So(properties, ShouldNotBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("with multiple IDs", func() {
			ids := []uint64{5, 256}

			Convey("and the DB returns valid rows", func() {
				rows := &mocks.Rows{}
				rows.On("Next", Anything).Return(true).Twice()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(nil)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				channelQuery, err := BuildQuery(ids, nil)
				So(err, ShouldBeNil)

				db.On("Query", ctx, "channel_properties_bulk", channelQuery.sql, channelQuery.params).Return(rows, nil)

				properties, err := cpr.GetAllChannelPropertiesBulk(ctx, ids, nil, util.ReadOptions{})
				So(properties, ShouldNotBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("with multiple login names and IDs", func() {
			logins := []string{"dadbot", "creative"}
			ids := []uint64{5, 256}

			Convey("and the DB returns valid rows", func() {
				rows := &mocks.Rows{}
				rows.On("Next", Anything).Return(true).Twice()
				rows.On("Scan", Anything, Anything, Anything, Anything, Anything, Anything, Anything, Anything).Return(nil)
				rows.On("Next", Anything).Return(false)
				rows.On("Close", Anything).Return(nil)
				rows.On("Err", Anything).Return(nil)

				channelQuery, err := BuildQuery(ids, logins)
				So(err, ShouldBeNil)

				db.On("Query", ctx, "channel_properties_bulk", channelQuery.sql, channelQuery.params).Return(rows, nil)

				properties, err := cpr.GetAllChannelPropertiesBulk(ctx, ids, logins, util.ReadOptions{})
				So(properties, ShouldNotBeNil)
				So(err, ShouldBeNil)
			})
		})
	})
}
