package channels

import (
	"time"

	"golang.org/x/net/context"
	"golang.org/x/sync/errgroup"

	"fmt"
	"strconv"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/web/users-service/backend"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/internal/utils"
	"code.justin.tv/web/users-service/models"
)

type cachedChannels struct {
	repo   Backend
	cacher backend.Cacher
}

func NewCachedBackend(repo Backend, cacher backend.Cacher) (Backend, error) {
	return &cachedChannels{
		repo:   repo,
		cacher: cacher,
	}, nil
}

func (c *cachedChannels) ExpireChannelProperties(ctx context.Context, prop *models.ChannelProperties) error {
	return c.cacher.ExpireProperties(ctx, prop)
}

func (c *cachedChannels) GetAllChannelPropertiesBulk(ctx context.Context, channelIDs []uint64, channelNames []string, opts util.ReadOptions) (props []models.ChannelProperties, err error) {
	start := time.Now()
	var cacheErr error
	defer func() {
		dur := time.Since(start)
		backend.ReportBulkIdentifiers(ctx, "BulkChannelID", len(channelIDs), dur, err)
		backend.ReportBulkIdentifiers(ctx, "BulkChannelName", len(channelNames), dur, err)
		backend.ReportBulkIdentifiers(ctx, "BulkChannel", len(channelNames)+len(channelIDs), dur, err)
	}()

	var missingIDs []uint64
	var missingNames []string

	if opts.ReadFromMaster {
		missingIDs = append(missingIDs, channelIDs...)
		missingNames = append(missingNames, channelNames...)
	} else {
		var g errgroup.Group
		var ids []models.ChannelProperties
		var names []models.ChannelProperties

		g.Go(func() error {
			if len(channelIDs) == 0 {
				return nil
			}

			keys := make([]string, len(channelIDs))
			for i, id := range channelIDs {
				keys[i] = models.ChannelIDToString(id)
			}

			missingIdx, cacheErr := c.cacher.BulkGetProperties(ctx, models.ChannelsIDCacheKey, keys, &ids)
			if cacheErr != nil {
				if !utils.IsKnownErr(cacheErr) {
					logx.Error(ctx, "failed to bulk get channel ids", logx.Fields{
						"error": cacheErr.Error(),
						"ids":   len(channelIDs),
					})
				}
				missingIDs = append(missingIDs, channelIDs...)
				return err
			}

			for _, missing := range missingIdx {
				missingIDs = append(missingIDs, channelIDs[missing])
			}

			return nil
		})

		g.Go(func() error {
			if len(channelNames) == 0 {
				return nil
			}

			missingIdx, err := c.cacher.BulkGetProperties(ctx, models.ChannelsLoginCacheKey, channelNames, &names)
			if err != nil {
				if !utils.IsKnownErr(err) {
					logx.Error(ctx, "failed to bulk get channel names", logx.Fields{
						"error": err.Error(),
						"names": len(channelNames),
					})
				}
				missingNames = append(missingNames, channelNames...)
				return err
			}

			for _, missing := range missingIdx {
				missingNames = append(missingNames, channelNames[missing])
			}
			return nil
		})

		// Errors are logged in functions and identifiers are added as missing
		xErr := g.Wait()
		if xErr != nil {
			// do nothing
		}

		props = append(props, ids...)
		props = append(props, names...)
	}
	if len(missingIDs) == 0 && len(missingNames) == 0 {
		return props, nil
	}

	var dbProps []models.ChannelProperties
	dbProps, err = c.repo.GetAllChannelPropertiesBulk(ctx, missingIDs, missingNames, opts)
	if err != nil && !utils.IsKnownErr(err) {
		return nil, errx.New(err, errx.Fields{
			"names": len(missingNames),
			"ids":   len(missingIDs),
		})
	}

	if len(dbProps) > 0 && !utils.IsHystrixErr(cacheErr) {
		bgCtx := backend.DetachContext(ctx)
		go func() {
			if cacheErr := c.cacher.BulkSetProperties(bgCtx, opts.OverwriteCache, models.ChannelPropertiesIterator(dbProps)); cacheErr != nil && !utils.IsKnownErr(cacheErr) {
				logx.Error(ctx, "failed to overwrite channel properties in cache", logx.Fields{
					"error": cacheErr.Error(),
					"props": len(dbProps),
				})
			}
		}()
	}

	props = append(props, dbProps...)

	return props, nil
}

func (c *cachedChannels) UpdateChannelProperties(ctx context.Context, updateProps models.UpdateChannelProperties) error {
	err := c.repo.UpdateChannelProperties(ctx, updateProps)
	if err != nil {
		return errx.New(err)
	}

	var prop models.ChannelProperties
	err = c.cacher.GetProperties(ctx, models.ChannelsIDCacheKey, models.ChannelIDToString(updateProps.ID), &prop)
	if backend.IsCacheMissErr(err) {
		return nil
	} else if err != nil {
		return err
	} else {
		return c.cacher.ExpireProperties(ctx, prop)
	}
}

func (c *cachedChannels) UpdateBasicChannelProperties(ctx context.Context, updateProps models.UpdateChannelProperties) error {
	err := c.repo.UpdateBasicChannelProperties(ctx, updateProps)
	if err != nil {
		return errx.New(err)
	}

	var prop models.ChannelProperties
	err = c.cacher.GetProperties(ctx, models.ChannelsIDCacheKey, models.ChannelIDToString(updateProps.ID), &prop)
	if backend.IsCacheMissErr(err) {
		return nil
	} else if err != nil {
		return err
	} else {
		return c.cacher.ExpireProperties(ctx, prop)
	}
}

func (c *cachedChannels) DeleteChannel(ctx context.Context, channelID string) error {
	if err := c.repo.DeleteChannel(ctx, channelID); err != nil {
		return err
	}

	id, err := strconv.ParseUint(channelID, 10, 64)
	if err != nil {
		return err
	}

	props, err := c.GetAllChannelPropertiesBulk(ctx, []uint64{id}, []string{}, util.ReadOptions{ReadFromMaster: true, OverwriteCache: true})
	if err != nil {
		return err
	}

	if len(props) != 1 {
		return errx.New(fmt.Sprintf("channel cannot be found for %s during overwrite cache", channelID))
	}

	return nil
}

func (c *cachedChannels) GetChannelsByRedirectChannel(ctx context.Context, name string) ([]models.ChannelProperties, error) {
	return c.repo.GetChannelsByRedirectChannel(ctx, name)
}
