package channels_test

import (
	"context"
	"errors"
	"testing"

	"github.com/stretchr/testify/mock"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/backend/channels"
	"code.justin.tv/web/users-service/backend/channels/mocks"
	backendMocks "code.justin.tv/web/users-service/backend/mocks"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/models"
)

func TestGetAllChannelPropertiesBulk(t *testing.T) {
	var repo *mocks.Backend
	var cacher *backendMocks.Cacher
	var uot channels.Backend

	var err error

	for name, scenario := range map[string]struct {
		IDs      []uint64
		Names    []string
		RO       util.ReadOptions
		Pre      func()
		Expected int
		Failure  bool
	}{
		"id and name cache miss": {
			IDs:      []uint64{1},
			Names:    []string{"asdf"},
			Expected: 2,
			Pre: func() {
				cacher.On("BulkGetProperties", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return([]int{0}, nil)

				repo.On("GetAllChannelPropertiesBulk", mock.Anything, []uint64{1}, []string{"asdf"}, mock.Anything).Return([]models.ChannelProperties{{}, {}}, nil)

				cacher.On("BulkSetProperties", mock.Anything, false, mock.Anything).Return(nil).Times(1)
			},
		},
		"id and name cache hit": {
			IDs:      []uint64{1},
			Names:    []string{"asdf"},
			Expected: 2,
			Pre: func() {
				idCall := cacher.On("BulkGetProperties", mock.Anything, mock.Anything, []string{"1"}, mock.Anything).Return(nil, nil)
				idCall.RunFn = func(args mock.Arguments) {
					arr := args[3].(*[]models.ChannelProperties)
					*arr = append(*arr, models.ChannelProperties{})
				}

				nameCall := cacher.On("BulkGetProperties", mock.Anything, mock.Anything, []string{"asdf"}, mock.Anything).Return(nil, nil)
				nameCall.RunFn = func(args mock.Arguments) {
					arr := args[3].(*[]models.ChannelProperties)
					*arr = append(*arr, models.ChannelProperties{})
				}
			},
		},
		"id and name cache hit and misses": {
			IDs:      []uint64{1, 2},
			Names:    []string{"asdf", "fdsa"},
			Expected: 4,
			Pre: func() {
				idCall := cacher.On("BulkGetProperties", mock.Anything, mock.Anything, []string{"1", "2"}, mock.Anything).Return([]int{1}, nil)
				idCall.RunFn = func(args mock.Arguments) {
					arr := args[3].(*[]models.ChannelProperties)
					*arr = append(*arr, models.ChannelProperties{})
				}

				nameCall := cacher.On("BulkGetProperties", mock.Anything, mock.Anything, []string{"asdf", "fdsa"}, mock.Anything).Return([]int{1}, nil)
				nameCall.RunFn = func(args mock.Arguments) {
					arr := args[3].(*[]models.ChannelProperties)
					*arr = append(*arr, models.ChannelProperties{})
				}

				repo.On("GetAllChannelPropertiesBulk", mock.Anything, []uint64{2}, []string{"fdsa"}, mock.Anything).Return([]models.ChannelProperties{{}, {}}, nil)

				cacher.On("BulkSetProperties", mock.Anything, false, mock.Anything).Return(nil).Times(1)
			},
		},
		"id and name cache miss and repo failure": {
			IDs:     []uint64{1, 2},
			Names:   []string{"asdf", "fdsa"},
			Failure: true,
			Pre: func() {
				cacher.On("BulkGetProperties", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return([]int{0, 1}, nil)

				repo.On("GetAllChannelPropertiesBulk", mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(nil, errors.New("fail"))
			},
		},
		"id !readfrommaster": {
			IDs: []uint64{1},
			RO: util.ReadOptions{
				ReadFromMaster: true,
				OverwriteCache: true,
			},
			Expected: 1,
			Pre: func() {
				repo.On("GetAllChannelPropertiesBulk", mock.Anything, []uint64{1}, []string(nil), util.ReadOptions{
					ReadFromMaster: true,
					OverwriteCache: true,
				}).Return([]models.ChannelProperties{{}}, nil)

				cacher.On("BulkSetProperties", mock.Anything, true, mock.Anything).Return(nil).Times(1)
			},
		},
	} {
		repo = &mocks.Backend{}
		cacher = &backendMocks.Cacher{}

		uot, err = channels.NewCachedBackend(repo, cacher)
		if err != nil {
			t.Fatal(err)
		}

		t.Run(name, func(t *testing.T) {
			scenario.Pre()

			properties, err := uot.GetAllChannelPropertiesBulk(context.Background(), scenario.IDs, scenario.Names, scenario.RO)
			if err != nil && !scenario.Failure {
				t.Fatal("unexpected failure:", err)
			} else if err == nil && scenario.Failure {
				t.Fatal("expected failure")
			}

			if len(properties) != scenario.Expected {
				t.Fatalf("expected %d properties, got %d", scenario.Expected, len(properties))
			}

			repo.AssertExpectations(t)
			// Not asserting expectations against the cacher because BulkSetProperties is run in a go routine
		})
	}
}

func TestDeleteChannel(t *testing.T) {
	var repo *mocks.Backend
	var cacher *backendMocks.Cacher
	var uot channels.Backend

	var err error

	for name, scenario := range map[string]struct {
		ID      string
		Pre     func()
		Failure bool
	}{
		"happy path": {
			ID: "1",
			Pre: func() {
				repo.On("DeleteChannel", mock.Anything, "1").Return(nil)
				cacher.On("BulkSetProperties", mock.Anything, true, mock.Anything).Return(nil).Times(1)
				repo.On("GetAllChannelPropertiesBulk", mock.Anything, []uint64{1}, mock.Anything, util.ReadOptions{ReadFromMaster: true, OverwriteCache: true}).Return([]models.ChannelProperties{{}}, nil)
			},
			Failure: false,
		},
		"overwrite cache failure with no record found": {
			ID: "1",
			Pre: func() {
				repo.On("DeleteChannel", mock.Anything, "1").Return(nil)
				cacher.On("BulkSetProperties", mock.Anything, true, mock.Anything).Return(nil).Times(1)
				repo.On("GetAllChannelPropertiesBulk", mock.Anything, []uint64{1}, mock.Anything, util.ReadOptions{ReadFromMaster: true, OverwriteCache: true}).Return([]models.ChannelProperties{}, nil)
			},
			Failure: true,
		},
		"overwrite cache failure with error": {
			ID: "1",
			Pre: func() {
				repo.On("DeleteChannel", mock.Anything, "1").Return(nil)
				cacher.On("BulkSetProperties", mock.Anything, true, mock.Anything).Return(nil).Times(1)
				repo.On("GetAllChannelPropertiesBulk", mock.Anything, []uint64{1}, mock.Anything, util.ReadOptions{ReadFromMaster: true, OverwriteCache: true}).Return([]models.ChannelProperties{{}}, errx.New("bad cache connection"))
			},
			Failure: true,
		},
	} {
		repo = &mocks.Backend{}
		cacher = &backendMocks.Cacher{}

		uot, err = channels.NewCachedBackend(repo, cacher)
		if err != nil {
			t.Fatal(err)
		}

		t.Run(name, func(t *testing.T) {
			scenario.Pre()

			err := uot.DeleteChannel(context.Background(), scenario.ID)
			if err != nil && !scenario.Failure {
				t.Fatal("unexpected failure:", err)
			} else if err == nil && scenario.Failure {
				t.Fatal("expected failure")
			}

			repo.AssertExpectations(t)
		})
	}
}
