package channels

import (
	"database/sql"

	"golang.org/x/net/context"

	"fmt"

	"strings"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/common/config"
	"code.justin.tv/common/yimg"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/database"
	"code.justin.tv/web/users-service/internal/image"
	"code.justin.tv/web/users-service/models"
)

const (
	sqlSelectPropertiesBulkPartial = `SELECT ` +
		` u.id,` +
		` "login",` +
		` COALESCE(directory_hidden, false) as "directory_hidden",` +
		` "broadcaster",` +
		` "broadcaster_language",` +
		` "broadcaster_software",` +
		` u.created_on,` +
		` "game",` +
		` "game_id",` +
		` "mature",` +
		` "status",` +
		` "title",` +
		` u.updated_on,` +
		` "views_count",` +
		` uap.block_non_public_ads,` +
		` uap.prerolls_disabled,` +
		` uap.postrolls_disabled,` +
		` uap.fight_ad_block,` +
		` ubp.last_broadcast_time,` +
		` ubp.last_broadcast_id,` +
		` ubp.last_live_notification_sent,` +
		` ucp.about,` +
		` ucp.redirect_channel,` +
		` ucp.primary_team_id,` +
		` ucp.disable_chat,` +
		` uip.channel_offline_image,` +
		` uip.profile_banner,` +
		` uip.profile_banner_background_color,` +
		` u.dmca_violation,` +
		` u.terms_of_service_violation,` +
		` u.deleted_on` +
		` FROM users u` +
		` LEFT JOIN user_ad_properties uap on (u.id = uap.user_id)` +
		` LEFT JOIN user_broadcast_properties ubp on (u.id = ubp.user_id)` +
		` LEFT JOIN user_channel_properties ucp on (u.id = ucp.user_id)` +
		` LEFT JOIN user_image_properties uip on (u.id = uip.user_id)` +
		` WHERE`

	sqlFindRedirectChannelDuringRename = `SELECT user_id FROM user_channel_properties WHERE LOWER(redirect_channel) = $1`
)

func GetAllChannelPropertiesFromRows(rows database.Rows) ([]models.ChannelProperties, error) {
	var offlineImage, profileBanner *string
	results := []models.ChannelProperties{}

	for rows.Next() {
		cp := &models.ChannelProperties{}
		err := rows.Scan(
			&cp.ID,
			&cp.Name,
			&cp.DirectoryHidden,
			&cp.Broadcaster,
			&cp.BroadcasterLanguage,
			&cp.BroadcasterSoftware,
			&cp.CreatedOn,
			&cp.Game,
			&cp.GameID,
			&cp.Mature,
			&cp.Status,
			&cp.Title,
			&cp.UpdatedOn,
			&cp.ViewsCount,

			&cp.BlockNonPublicAds,
			&cp.PrerollsDisabled,
			&cp.PostrollsDisabled,
			&cp.FightAdBlock,

			&cp.LastBroadcastTime,
			&cp.LastBroadcastID,
			&cp.LastLiveNotificationSent,

			&cp.About,
			&cp.RedirectChannel,
			&cp.PrimaryTeamID,
			&cp.DisableChat,

			&offlineImage,
			&profileBanner,
			&cp.ProfileBannerBackgroundColor,
			&cp.DmcaViolation,
			&cp.TermsOfServiceViolation,
			&cp.DeletedOn)
		switch {
		case err == sql.ErrNoRows:
			return nil, errx.New(ErrNoChannelProperties)
		case err != nil:
			return nil, errx.New(err)
		default:
		}

		offlineImageProp, _, err := image.GetChannelOfflineImages(offlineImage, cp.Name)
		if err != nil {
			if metricErr := config.ClusterStatsd().Inc("parse.channel_offline_image.failure", 1, 0.1); metricErr != nil {
				logx.Warn(context.Background(), fmt.Sprintf("error incrementing image metrics: %s", metricErr))
			}
			cp.ChannelOfflineImage = yimg.Images{}
		} else if offlineImageProp != nil {
			cp.ChannelOfflineImage = *offlineImageProp
		}

		profileBannerProp, _, err := image.GetProfileBanners(profileBanner, cp.Name)
		if err != nil {
			if metricErr := config.ClusterStatsd().Inc("parse.profile_banner.failure", 1, 0.1); metricErr != nil {
				logx.Warn(context.Background(), fmt.Sprintf("error incrementing image metrics: %s", metricErr))
			}
			cp.ProfileBanner = yimg.Images{}
		} else if profileBannerProp != nil {
			cp.ProfileBanner = *profileBannerProp
		}

		results = append(results, *cp)
	}

	return results, errx.New(rows.Err())
}

func (c *channelPropertiesRepoImpl) GetAllChannelPropertiesBulk(ctx context.Context, channelIDs []uint64, channelNames []string, _ util.ReadOptions) (props []models.ChannelProperties, err error) {
	// TODO: Update channels backend to treat ids as string
	channelQuery, err := BuildQuery(channelIDs, channelNames)

	var rows database.Rows
	rows, err = c.sdb.Query(ctx, "channel_properties_bulk", channelQuery.sql, channelQuery.params...)
	if err != nil {
		return
	}

	props, err = GetAllChannelPropertiesFromRows(rows)
	return
}

func (c *channelPropertiesRepoImpl) GetChannelsByRedirectChannel(ctx context.Context, name string) ([]models.ChannelProperties, error) {
	props := []models.ChannelProperties{}
	var rows database.Rows
	rows, err := c.sdb.Query(ctx, "get_channels_by_redirect_channel", sqlFindRedirectChannelDuringRename, strings.ToLower(name))
	if err != nil {
		return props, err
	}

	for rows.Next() {
		cp := &models.ChannelProperties{}
		err := rows.Scan(
			&cp.ID,
		)
		switch {
		case err == sql.ErrNoRows:
			return nil, nil
		case err != nil:
			return nil, errx.New(err)
		default:
		}

		props = append(props, *cp)
	}

	return props, nil
}
