package channels

import (
	"fmt"

	"golang.org/x/net/context"

	"github.com/afex/hystrix-go/hystrix"

	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/web/users-service/backend/util"
	"code.justin.tv/web/users-service/models"
)

func init() {
	// These values were determined by looking at the past 7 days of database rps and p99s.
	// The graphs are in the "Database" section of https://grafana.internal.justin.tv/dashboard/db/users-service-v2?refresh=30s&orgId=1&from=now-7d&to=now-1m.
	// The max concurrent requests was determined by going above the max of the past week / divided by hosts at the time.
	hystrix.Configure(map[string]hystrix.CommandConfig{
		"Channels.GetAllChannelPropertiesBulk": {
			Timeout:               5000,
			MaxConcurrentRequests: 5000,
		},
		"Channels.UpdateChannelProperties": {
			Timeout:               5000,
			MaxConcurrentRequests: 20,
		},
		"Channels.UpdateBasicChannelProperties": {
			Timeout:               5000,
			MaxConcurrentRequests: 10,
		},
		"Channels.DeleteChannel": {
			Timeout: 5000,
		},
		"Channels.ExpireChannelProperties": {
			Timeout:               5000,
			MaxConcurrentRequests: 20,
		},
	})
}

type HystrixBackend struct {
	Backend Backend
}

var _ Backend = (*HystrixBackend)(nil)

func skipError(err error) bool {
	err = errx.Unwrap(err)
	return err == ErrNoChannelProperties
}

func (b *HystrixBackend) ExpireChannelProperties(ctx context.Context, prop *models.ChannelProperties) error {
	var err error

	hystrixErr := hystrix.Do("Channels.ExpireChannelProperties", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Channels.ExpireChannelProperties circuit panic=%v", r))
			}
		}()

		err = b.Backend.ExpireChannelProperties(ctx, prop)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}

func (b *HystrixBackend) GetAllChannelPropertiesBulk(ctx context.Context, channelIDs []uint64, channelNames []string, opts util.ReadOptions) ([]models.ChannelProperties, error) {
	var ret []models.ChannelProperties
	var err error

	hystrixErr := hystrix.Do("Channels.GetAllChannelPropertiesBulk", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Channels.GetAllChannelPropertiesBulk circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetAllChannelPropertiesBulk(ctx, channelIDs, channelNames, opts)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}

func (b *HystrixBackend) UpdateChannelProperties(ctx context.Context, updateProps models.UpdateChannelProperties) error {
	var err error

	hystrixErr := hystrix.Do("Channels.UpdateChannelProperties", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Channels.UpdateChannelProperties circuit panic=%v", r))
			}
		}()

		err = b.Backend.UpdateChannelProperties(ctx, updateProps)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}

func (b *HystrixBackend) UpdateBasicChannelProperties(ctx context.Context, updateProps models.UpdateChannelProperties) error {
	var err error

	hystrixErr := hystrix.Do("Channels.UpdateBasicChannelProperties", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Channels.UpdateBasicChannelProperties circuit panic=%v", r))
			}
		}()

		err = b.Backend.UpdateBasicChannelProperties(ctx, updateProps)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}

func (b *HystrixBackend) DeleteChannel(ctx context.Context, channelID string) error {
	var err error

	hystrixErr := hystrix.Do("Channels.DeleteChannel", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Channels.DeleteChannel circuit panic=%v", r))
			}
		}()

		err = b.Backend.DeleteChannel(ctx, channelID)
		return err
	}, nil)

	if hystrixErr != nil {
		return hystrixErr
	}

	return err
}

func (b *HystrixBackend) GetChannelsByRedirectChannel(ctx context.Context, name string) ([]models.ChannelProperties, error) {
	var ret []models.ChannelProperties
	var err error

	hystrixErr := hystrix.Do("Channels.GetChannelsByRedirectChannel", func() (rErr error) {
		defer func() {
			if r := recover(); r != nil {
				rErr = errx.New(fmt.Sprintf("Channels.GetChannelsByRedirectChannel circuit panic=%v", r))
			}
		}()

		ret, err = b.Backend.GetChannelsByRedirectChannel(ctx, name)
		if skipError(err) {
			return nil
		}
		return err
	}, nil)

	if hystrixErr != nil {
		return nil, hystrixErr
	}

	return ret, err
}
