package channels_test

import (
	"testing"

	"errors"

	. "code.justin.tv/web/users-service/backend/channels"
	"code.justin.tv/web/users-service/database/mocks"
	. "code.justin.tv/web/users-service/internal/testutils"
	"code.justin.tv/web/users-service/models"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"
	"golang.org/x/net/context"
)

type UpdateChannelBasicPropertiesTest struct {
	suite.Suite
	sdb     *mocks.Querier
	mdb     *mocks.Querier
	tx      *mocks.Tx
	backend Backend
	ctx     context.Context
	upProp  models.UpdateChannelProperties
}

func (suite *UpdateChannelBasicPropertiesTest) SetupTest() {
	var err error
	suite.sdb = &mocks.Querier{}
	suite.mdb = &mocks.Querier{}
	suite.tx = &mocks.Tx{}
	suite.backend, err = NewBackend(suite.sdb, suite.mdb)
	assert.NoError(suite.T(), err)
	suite.ctx = context.Background()
	suite.upProp = models.UpdateChannelProperties{
		ID:                  Channels[AdminUserID].ID,
		DirectoryHidden:     GetBoolPointer(true),
		BroadcasterLanguage: GetStringPointer("ko"),
		BroadcasterSoftware: GetStringPointer("bala"),
		Game:                GetStringPointer("overwatch"),
		GameID:              nil,
		Mature:              GetBoolPointer(true),
		Status:              nil,
		Title:               nil,
		UpdatedOn:           nil,
		ViewsCount:          nil,
	}
}

func (suite *UpdateChannelBasicPropertiesTest) TestUpdateChannel() {
	suite.tx.Mock.On("Exec", mock.Anything, "channel_properties_update", SqlUpdateChannelBasicProperties, []interface{}{
		suite.upProp.ID, suite.upProp.DirectoryHidden, suite.upProp.Broadcaster, suite.upProp.BroadcasterLanguage, suite.upProp.BroadcasterSoftware, suite.upProp.Game, suite.upProp.GameID,
		suite.upProp.Mature, suite.upProp.Status, suite.upProp.Title, suite.upProp.ViewsCount,
	}).Return(nil, nil)

	err := UpdateChannelBasicProperties(suite.ctx, suite.upProp, suite.tx)
	assert.NoError(suite.T(), err)
	suite.tx.AssertExpectations(suite.T())
}

func (suite *UpdateChannelBasicPropertiesTest) TestUpdateChannelDBError() {
	suite.tx.Mock.On("Exec", mock.Anything, "channel_properties_update", SqlUpdateChannelBasicProperties, []interface{}{
		suite.upProp.ID, suite.upProp.DirectoryHidden, suite.upProp.Broadcaster, suite.upProp.BroadcasterLanguage, suite.upProp.BroadcasterSoftware, suite.upProp.Game, suite.upProp.GameID,
		suite.upProp.Mature, suite.upProp.Status, suite.upProp.Title, suite.upProp.ViewsCount,
	}).Return(nil, errors.New(""))

	err := UpdateChannelBasicProperties(suite.ctx, suite.upProp, suite.tx)
	assert.Error(suite.T(), err)
}

func TestUpdateChannelBasicPropertiesSuite(t *testing.T) {
	suite.Run(t, new(UpdateChannelBasicPropertiesTest))
}
